"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findIntegration = exports.CUSTOM_ASSETS_PREFIX = void 0;
exports.getComponentTemplate = getComponentTemplate;
exports.getCustomAssets = void 0;
exports.getPipeline = getPipeline;
exports.getPipelinesFromVars = getPipelinesFromVars;
exports.installCustomAsset = installCustomAsset;
var _lodash = require("lodash");
var _retry = require("../../services/epm/elasticsearch/retry");
var _services = require("../../services");
var _constants = require("../../constants");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DELETED_ASSET_TTL = 7 * 24 * 60 * 60 * 1000; // 7 days
const CUSTOM_ASSETS_PREFIX = exports.CUSTOM_ASSETS_PREFIX = '*@custom';
const findIntegration = (assetName, integrations) => {
  const matches = assetName.match(/^(\w*)?(?:\-)?(\w*)(?:\-)?(?:\.)?(?:\w*)?@custom$/);
  if (!matches) return undefined;
  return integrations.find(integration => {
    return ['logs', 'metrics', 'traces'].includes(matches[1]) && matches[2] === integration.package_name ||
    // e.g. logs-system.auth@custom
    !['logs', 'metrics', 'traces'].includes(matches[1]) &&
    // e.g. synthetics-tcp@custom
    matches[1] === integration.package_name;
  });
};
exports.findIntegration = findIntegration;
function getComponentTemplate(esClient, name, abortController) {
  return esClient.cluster.getComponentTemplate({
    name
  }, {
    ignore: [404],
    signal: abortController.signal
  });
}
function getPipeline(esClient, name, abortController) {
  return esClient.ingest.getPipeline({
    id: name
  }, {
    ignore: [404],
    signal: abortController.signal
  });
}
const getCustomAssets = async (esClient, soClient, integrations, abortController, previousSyncIntegrationsData) => {
  const customTemplates = await getComponentTemplate(esClient, CUSTOM_ASSETS_PREFIX, abortController);
  const customAssetsComponentTemplates = customTemplates.component_templates.reduce((acc, template) => {
    var _integration$package_, _integration$package_2;
    const integration = findIntegration(template.name, integrations);
    if (!integration) return acc;
    acc.push({
      type: 'component_template',
      name: template.name,
      package_name: (_integration$package_ = integration.package_name) !== null && _integration$package_ !== void 0 ? _integration$package_ : '',
      package_version: (_integration$package_2 = integration.package_version) !== null && _integration$package_2 !== void 0 ? _integration$package_2 : '',
      is_deleted: false,
      template: template.component_template.template
    });
    return acc;
  }, []);
  const ingestPipelines = await getPipeline(esClient, CUSTOM_ASSETS_PREFIX, abortController);
  const customAssetsIngestPipelines = Object.keys(ingestPipelines).reduce((acc, pipeline) => {
    var _integration$package_3, _integration$package_4;
    const integration = findIntegration(pipeline, integrations);
    if (!integration) return acc;
    acc.push({
      type: 'ingest_pipeline',
      name: pipeline,
      package_name: (_integration$package_3 = integration.package_name) !== null && _integration$package_3 !== void 0 ? _integration$package_3 : '',
      package_version: (_integration$package_4 = integration.package_version) !== null && _integration$package_4 !== void 0 ? _integration$package_4 : '',
      is_deleted: false,
      pipeline: ingestPipelines[pipeline]
    });
    return acc;
  }, []);
  const customPipelineFromVars = await getPipelinesFromVars(esClient, soClient, abortController);
  const updatedAssets = [...customAssetsComponentTemplates, ...customAssetsIngestPipelines, ...customPipelineFromVars];
  const deletedAssets = updateDeletedAssets(previousSyncIntegrationsData, updatedAssets);
  return [...updatedAssets, ...deletedAssets];
};
exports.getCustomAssets = getCustomAssets;
async function getPipelinesFromVars(esClient, soClient, abortController) {
  const packagePolicies = await _services.packagePolicyService.list(soClient, {
    perPage: _constants.SO_SEARCH_LIMIT,
    spaceId: '*'
  });
  const customPipelineFromVars = [];
  for (const packagePolicy of packagePolicies.items) {
    for (const input of packagePolicy.inputs) {
      for (const stream of input.streams) {
        var _stream$vars;
        // find stream vars called `pipeline`
        if ((_stream$vars = stream.vars) !== null && _stream$vars !== void 0 && _stream$vars.pipeline && stream.vars.pipeline.value) {
          const pipelineName = stream.vars.pipeline.value;
          // find pipeline definition for the matching var value
          const pipelineDef = await getPipeline(esClient, pipelineName, abortController);
          if (pipelineDef[pipelineName]) {
            var _packagePolicy$packag, _packagePolicy$packag2, _packagePolicy$packag3, _packagePolicy$packag4;
            customPipelineFromVars.push({
              type: 'ingest_pipeline',
              name: pipelineName,
              package_name: (_packagePolicy$packag = (_packagePolicy$packag2 = packagePolicy.package) === null || _packagePolicy$packag2 === void 0 ? void 0 : _packagePolicy$packag2.name) !== null && _packagePolicy$packag !== void 0 ? _packagePolicy$packag : '',
              package_version: (_packagePolicy$packag3 = (_packagePolicy$packag4 = packagePolicy.package) === null || _packagePolicy$packag4 === void 0 ? void 0 : _packagePolicy$packag4.version) !== null && _packagePolicy$packag3 !== void 0 ? _packagePolicy$packag3 : '',
              is_deleted: false,
              pipeline: pipelineDef[pipelineName]
            });
          }
        }
      }
    }
  }
  return customPipelineFromVars;
}
function updateDeletedAssets(previousSyncIntegrationsData, updatedAssets) {
  var _previousSyncIntegrat;
  const deletedAssets = [];
  Object.values((_previousSyncIntegrat = previousSyncIntegrationsData === null || previousSyncIntegrationsData === void 0 ? void 0 : previousSyncIntegrationsData.custom_assets) !== null && _previousSyncIntegrat !== void 0 ? _previousSyncIntegrat : {}).forEach(existingAsset => {
    if (existingAsset.is_deleted) {
      if (existingAsset.deleted_at && Date.now() - Date.parse(existingAsset.deleted_at) < DELETED_ASSET_TTL) {
        deletedAssets.push(existingAsset);
      }
    } else {
      const matchingAsset = updatedAssets.find(asset => existingAsset.name === asset.name && existingAsset.type === asset.type);
      if (!matchingAsset) {
        deletedAssets.push({
          ...existingAsset,
          is_deleted: true,
          deleted_at: new Date().toISOString()
        });
      }
    }
  });
  return deletedAssets;
}
async function updateComponentTemplate(customAsset, esClient, abortController, logger) {
  var _customTemplates$comp;
  const customTemplates = await getComponentTemplate(esClient, customAsset.name, abortController);
  const existingTemplate = (_customTemplates$comp = customTemplates.component_templates) === null || _customTemplates$comp === void 0 ? void 0 : _customTemplates$comp.find(template => template.name === customAsset.name);
  if (customAsset.is_deleted) {
    if (existingTemplate) {
      logger.debug(`Deleting component template: ${customAsset.name}`);
      return (0, _retry.retryTransientEsErrors)(() => esClient.cluster.deleteComponentTemplate({
        name: customAsset.name
      }, {
        signal: abortController.signal
      }), {
        logger
      });
    } else {
      return;
    }
  }
  let shouldUpdateTemplate = false;
  if (existingTemplate) {
    shouldUpdateTemplate = !(0, _lodash.isEqual)(existingTemplate.component_template.template, customAsset.template);
  } else {
    shouldUpdateTemplate = true;
  }
  if (shouldUpdateTemplate) {
    logger.debug(`Updating component template: ${customAsset.name}`);
    return (0, _retry.retryTransientEsErrors)(() => esClient.cluster.putComponentTemplate({
      name: customAsset.name,
      template: customAsset.template
    }, {
      signal: abortController.signal
    }), {
      logger
    });
  }
}
async function updateIngestPipeline(customAsset, esClient, abortController, logger) {
  const ingestPipelines = await getPipeline(esClient, customAsset.name, abortController);
  const existingPipeline = ingestPipelines[customAsset.name];
  if (customAsset.is_deleted) {
    if (existingPipeline) {
      logger.debug(`Deleting ingest pipeline: ${customAsset.name}`);
      return (0, _retry.retryTransientEsErrors)(() => esClient.ingest.deletePipeline({
        id: customAsset.name
      }, {
        signal: abortController.signal
      }), {
        logger
      });
    } else {
      return;
    }
  }

  // Remove system-managed properties (dates) that cannot be set during create/update of ingest pipelines
  const customAssetPipelineWithoutTimestamps = (0, _lodash.omit)(customAsset.pipeline, ['created_date', 'created_date_millis', 'modified_date', 'modified_date_millis']);
  const existingPipelineWithoutTimestamps = (0, _lodash.omit)(existingPipeline, ['created_date', 'created_date_millis', 'modified_date', 'modified_date_millis']);
  let shouldUpdatePipeline = false;
  if (existingPipeline) {
    shouldUpdatePipeline = existingPipeline.version && existingPipeline.version < customAsset.pipeline.version || !existingPipeline.version && !(0, _lodash.isEqual)(existingPipelineWithoutTimestamps, customAssetPipelineWithoutTimestamps);
  } else {
    shouldUpdatePipeline = true;
  }
  if (shouldUpdatePipeline && customAsset.pipeline.processors.some(p => p.enrich)) {
    throw new _errors.FleetError(`Syncing ingest pipelines that reference enrich policies is not supported. Please sync manually.`);
  }
  if (shouldUpdatePipeline) {
    logger.debug(`Updating ingest pipeline: ${customAsset.name}`);
    return (0, _retry.retryTransientEsErrors)(() => esClient.ingest.putPipeline({
      id: customAsset.name,
      ...customAssetPipelineWithoutTimestamps
    }, {
      signal: abortController.signal
    }), {
      logger
    });
  }
}
async function installCustomAsset(customAsset, esClient, abortController, logger) {
  if (customAsset.type === 'component_template') {
    return updateComponentTemplate(customAsset, esClient, abortController, logger);
  } else if (customAsset.type === 'ingest_pipeline') {
    return updateIngestPipeline(customAsset, esClient, abortController, logger);
  }
}