"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentlessPoliciesServiceImpl = void 0;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _spacesUtils = require("@kbn/spaces-utils");
var _constants = require("../../../common/constants");
var _simplified_package_policy_helper = require("../../../common/services/simplified_package_policy_helper");
var _http_authorization_header = require("../../../common/http_authorization_header");
var _agent_policy = require("../agent_policy");
var _packages = require("../epm/packages");
var _app_context = require("../app_context");
var _agentless_policy_helper = require("../../../common/services/agentless_policy_helper");
var _agentless_agent = require("../agents/agentless_agent");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAgentlessPolicy = packageInfo => {
  var _packageInfo$policy_t, _agentlessPolicyTempl;
  if (!(packageInfo !== null && packageInfo !== void 0 && packageInfo.policy_templates) && !(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t = packageInfo.policy_templates) !== null && _packageInfo$policy_t !== void 0 && _packageInfo$policy_t.some(policy => policy.deployment_modes))) {
    return;
  }
  const agentlessPolicyTemplate = packageInfo.policy_templates.find(policy => policy.deployment_modes);

  // assumes that all the policy templates agentless deployments modes indentify have the same organization, division and team
  const agentlessInfo = agentlessPolicyTemplate === null || agentlessPolicyTemplate === void 0 ? void 0 : (_agentlessPolicyTempl = agentlessPolicyTemplate.deployment_modes) === null || _agentlessPolicyTempl === void 0 ? void 0 : _agentlessPolicyTempl.agentless;
  if (!(agentlessInfo !== null && agentlessInfo !== void 0 && agentlessInfo.resources)) {
    return;
  }
  return {
    resources: agentlessInfo.resources
  };
};
class AgentlessPoliciesServiceImpl {
  constructor(packagePolicyService, soClient, esClient, logger) {
    this.packagePolicyService = packagePolicyService;
    this.soClient = soClient;
    this.esClient = esClient;
    this.logger = logger;
  }
  async createAgentlessPolicy(data, context, request) {
    const packagePolicyId = data.id || (0, _uuid.v4)();
    const agentPolicyId = packagePolicyId; // Use the same ID for agent policy and package policy
    const force = data.force;
    this.logger.debug('Creating agentless policy');
    const user = request ? _app_context.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined : undefined;
    const authorizationHeader = request ? _http_authorization_header.HTTPAuthorizationHeader.parseFromRequest(request, user === null || user === void 0 ? void 0 : user.username) : null;
    const spaceId = this.soClient.getCurrentNamespace() || _spacesUtils.DEFAULT_SPACE_ID;
    let createdAgentPolicyId;
    try {
      const pkg = data.package;
      this.logger.debug(`Creating agentless agent policy ${agentPolicyId}`);
      const pkgInfo = await (0, _packages.getPackageInfo)({
        savedObjectsClient: this.soClient,
        pkgName: pkg.name,
        pkgVersion: pkg.version,
        ignoreUnverified: force,
        prerelease: true
      });
      const {
        outputId,
        fleetServerId
      } = _agentless_agent.agentlessAgentService.getDefaultSettings();
      const agentPolicyName = (0, _agentless_policy_helper.getAgentlessAgentPolicyNameFromPackagePolicyName)(data.name);
      const agentPolicy = await _agent_policy.agentPolicyService.create(this.soClient, this.esClient, {
        name: agentPolicyName,
        description: 'Internal agentless policy',
        inactivity_timeout: _constants.AGENTLESS_AGENT_POLICY_INACTIVITY_TIMEOUT,
        supports_agentless: true,
        namespace: data.namespace || 'default',
        monitoring_enabled: [],
        keep_monitoring_alive: true,
        agentless: getAgentlessPolicy(pkgInfo),
        global_data_tags: (0, _agentless_policy_helper.getAgentlessGlobalDataTags)(pkgInfo),
        fleet_server_host_id: fleetServerId,
        data_output_id: outputId,
        is_protected: false
      }, {
        id: agentPolicyId,
        skipDeploy: true,
        authorizationHeader,
        user
      });
      createdAgentPolicyId = agentPolicy.id;
      const newPolicy = {
        ...(0, _lodash.omit)(data, 'id', 'package'),
        namespace: data.namespace || 'default',
        policy_ids: [agentPolicy.id],
        supports_agentless: true
      };
      const newPackagePolicy = (0, _simplified_package_policy_helper.simplifiedPackagePolicytoNewPackagePolicy)(newPolicy, pkgInfo);

      // Create package policy
      this.logger.debug(`Creating agentless package policy ${packagePolicyId}`);
      const packagePolicy = await this.packagePolicyService.create(this.soClient, this.esClient, newPackagePolicy, {
        id: packagePolicyId,
        force,
        bumpRevision: false,
        spaceId,
        authorizationHeader,
        user
      }, context, request);
      this.logger.debug(`Deploy agentless policy ${agentPolicyId}`);
      await _agent_policy.agentPolicyService.deployPolicy(this.soClient, agentPolicyId, undefined, {
        throwOnAgentlessError: true
      });
      return packagePolicy;
    } catch (err) {
      // If policy was created and error happens later during package policy creation or agentless API call, delete the created policy to avoid orphaned policies
      if (createdAgentPolicyId) {
        await _agent_policy.agentPolicyService.delete(this.soClient, this.esClient, createdAgentPolicyId, {
          force: true
        }).catch(e => {
          this.logger.error(`Failed to delete agentless agent policy ${createdAgentPolicyId}: ${e.message}`, {
            error: e
          });
        });
      }
      throw err;
    }
  }
  async deleteAgentlessPolicy(policyId, options, context, request) {
    this.logger.debug(`Deleting agentless policy ${policyId}`);
    const user = request ? _app_context.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined : undefined;
    const agentPolicy = await _agent_policy.agentPolicyService.get(this.soClient, policyId);
    if (!(agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless)) {
      throw new Error(`Policy ${policyId} is not an agentless policy`);
    }

    // Delete agent policy
    await _agent_policy.agentPolicyService.delete(this.soClient, this.esClient, policyId, {
      force: options === null || options === void 0 ? void 0 : options.force,
      user
    });
  }
}
exports.AgentlessPoliciesServiceImpl = AgentlessPoliciesServiceImpl;