"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSortFieldForAPI = void 0;
exports.useFetchAgentsData = useFetchAgentsData;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _reactRouterDom = require("react-router-dom");
var _reactQuery = require("@kbn/react-query");
var _services = require("../../../../../../../common/services");
var _hooks = require("../../../../hooks");
var _services2 = require("../../../../services");
var _constants = require("../../../../constants");
var _get_kuery = require("../utils/get_kuery");
var _use_session_agent_list_state = require("./use_session_agent_list_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const REFRESH_INTERVAL_MS = 30000;
const MAX_AGENT_ACTIONS = 100;
/** Allow to fetch full agent policy using a cache */
function useFullAgentPolicyFetcher() {
  const authz = (0, _hooks.useAuthz)();
  const fetchedAgentPoliciesRef = (0, _react.useRef)({});
  const fetchPolicies = (0, _react.useCallback)(async policiesIds => {
    const policiesToFetchIds = policiesIds.reduce((acc, policyId) => {
      if (!fetchedAgentPoliciesRef.current[policyId]) {
        acc.push(policyId);
      }
      return acc;
    }, []);
    if (policiesToFetchIds.length) {
      const bulkGetAgentPoliciesResponse = await (0, _hooks.sendBulkGetAgentPoliciesForRq)(policiesToFetchIds, {
        full: authz.fleet.readAgentPolicies,
        ignoreMissing: true
      });
      bulkGetAgentPoliciesResponse.items.forEach(agentPolicy => {
        fetchedAgentPoliciesRef.current[agentPolicy.id] = agentPolicy;
      });
    }
    return policiesIds.reduce((acc, policyId) => {
      if (fetchedAgentPoliciesRef.current[policyId]) {
        acc.push(fetchedAgentPoliciesRef.current[policyId]);
      }
      return acc;
    }, []);
  }, [authz.fleet.readAgentPolicies]);
  return (0, _react.useMemo)(() => ({
    fetchPolicies
  }), [fetchPolicies]);
}
const VERSION_FIELD = 'local_metadata.elastic.agent.version';
const HOSTNAME_FIELD = 'local_metadata.host.hostname';
const getSortFieldForAPI = field => {
  if ([VERSION_FIELD, HOSTNAME_FIELD].includes(field)) {
    return `${field}.keyword`;
  }
  return field;
};
exports.getSortFieldForAPI = getSortFieldForAPI;
function useFetchAgentsData() {
  const fullAgentPolicyFecher = useFullAgentPolicyFetcher();
  const {
    notifications
  } = (0, _hooks.useStartServices)();
  const history = (0, _reactRouterDom.useHistory)();
  const {
    urlParams,
    toUrlParams
  } = (0, _hooks.useUrlParams)();
  const {
    showAgentless
  } = (0, _hooks.useAgentlessResources)();
  const defaultKuery = urlParams.kuery || '';
  const urlHasInactive = urlParams.showInactive === 'true';
  const isUsingParams = defaultKuery || urlHasInactive;

  // Extract state from session storage hook
  const sessionState = (0, _use_session_agent_list_state.useSessionAgentListState)();
  const {
    search,
    selectedAgentPolicies,
    selectedStatus,
    selectedTags,
    showUpgradeable,
    sort,
    page,
    updateTableState
  } = sessionState;

  // If URL params are used, reset the table state to defaults with the param options
  (0, _react.useEffect)(() => {
    if (isUsingParams) {
      updateTableState({
        ..._use_session_agent_list_state.defaultAgentListState,
        search: defaultKuery,
        selectedStatus: [...new Set([...selectedStatus, ...(urlHasInactive ? ['inactive'] : [])])]
      });
    }
    // Empty array so that this only runs once on mount
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // Sync URL kuery param with session storage search to maintain shareable state
  (0, _react.useEffect)(() => {
    const currentUrlKuery = urlParams.kuery || '';
    // If search is empty and URL has kuery, or search differs from URL, update URL
    if (search === '' && currentUrlKuery !== '' || search && search !== currentUrlKuery) {
      const {
        kuery: _,
        ...restParams
      } = urlParams;
      const newParams = search === '' ? restParams : {
        ...restParams,
        kuery: search
      };
      history.replace({
        search: toUrlParams(newParams)
      });
    }
  }, [search, urlParams, history, toUrlParams]);

  // Flag to indicate if filters differ from default state
  const isUsingFilter = (0, _react.useMemo)(() => {
    return search !== _use_session_agent_list_state.defaultAgentListState.search || !(0, _lodash.isEqual)(selectedAgentPolicies, _use_session_agent_list_state.defaultAgentListState.selectedAgentPolicies) || !(0, _lodash.isEqual)(selectedStatus, _use_session_agent_list_state.defaultAgentListState.selectedStatus) || !(0, _lodash.isEqual)(selectedTags, _use_session_agent_list_state.defaultAgentListState.selectedTags) || showUpgradeable !== _use_session_agent_list_state.defaultAgentListState.showUpgradeable;
  }, [search, selectedAgentPolicies, selectedStatus, selectedTags, showUpgradeable]);

  // Create individual setters using updateTableState
  const setSearchState = (0, _react.useCallback)(value => updateTableState({
    search: value
  }), [updateTableState]);
  const setSelectedAgentPolicies = (0, _react.useCallback)(value => updateTableState({
    selectedAgentPolicies: value
  }), [updateTableState]);
  const setSelectedStatus = (0, _react.useCallback)(value => updateTableState({
    selectedStatus: value
  }), [updateTableState]);
  const setSelectedTags = (0, _react.useCallback)(value => updateTableState({
    selectedTags: value
  }), [updateTableState]);
  const setShowUpgradeable = (0, _react.useCallback)(value => updateTableState({
    showUpgradeable: value
  }), [updateTableState]);
  const pageSizeOptions = [5, 20, 50];

  // Sync draftKuery with session storage search
  const [draftKuery, setDraftKuery] = (0, _react.useState)(search);
  (0, _react.useEffect)(() => {
    setDraftKuery(search);
  }, [search]);
  const showInactive = (0, _react.useMemo)(() => {
    return selectedStatus.some(status => status === 'inactive') || selectedStatus.length === 0;
  }, [selectedStatus]);
  const includeUnenrolled = (0, _react.useMemo)(() => {
    return selectedStatus.some(status => status === 'unenrolled') || selectedStatus.length === 0;
  }, [selectedStatus]);
  const setSearch = (0, _react.useCallback)(newVal => {
    setSearchState(newVal);
    if (newVal.trim() === '' && !urlParams.kuery) {
      return;
    }
    if (urlParams.kuery !== newVal) {
      const {
        kuery: _,
        ...restParams
      } = urlParams;
      const newParams = newVal === '' ? restParams : {
        ...restParams,
        kuery: newVal
      };
      history.replace({
        search: toUrlParams(newParams)
      });
    }
  }, [setSearchState, urlParams, history, toUrlParams]);

  // filters kuery
  let kuery = (0, _react.useMemo)(() => {
    return (0, _get_kuery.getKuery)({
      search,
      selectedAgentPolicies,
      selectedTags,
      selectedStatus
    });
  }, [search, selectedAgentPolicies, selectedStatus, selectedTags]);
  kuery = includeUnenrolled && kuery ? `status:* AND (${kuery})` : includeUnenrolled ? `status:*` : kuery;
  const [allTags, setAllTags] = (0, _react.useState)();
  const [latestAgentActionErrors, setLatestAgentActionErrors] = (0, _react.useState)([]);
  const {
    data: actionErrors
  } = (0, _reactQuery.useQuery)({
    refetchInterval: REFRESH_INTERVAL_MS,
    queryKey: ['get-action-statuses'],
    initialData: [],
    queryFn: async () => {
      var _actionStatusResponse;
      const actionStatusResponse = await (0, _hooks.sendGetActionStatus)({
        latest: REFRESH_INTERVAL_MS + 5000,
        // avoid losing errors
        perPage: MAX_AGENT_ACTIONS
      });
      return ((_actionStatusResponse = actionStatusResponse.data) === null || _actionStatusResponse === void 0 ? void 0 : _actionStatusResponse.items.filter(action => {
        var _action$latestErrors$, _action$latestErrors;
        return (_action$latestErrors$ = (_action$latestErrors = action.latestErrors) === null || _action$latestErrors === void 0 ? void 0 : _action$latestErrors.length) !== null && _action$latestErrors$ !== void 0 ? _action$latestErrors$ : 0 > 1;
      }).map(action => action.actionId)) || [];
    }
  });
  (0, _react.useEffect)(() => {
    const allRecentActionErrors = [...new Set([...latestAgentActionErrors, ...actionErrors])];
    if (!(0, _lodash.isEqual)(latestAgentActionErrors, allRecentActionErrors)) {
      setLatestAgentActionErrors(allRecentActionErrors);
    }
  }, [latestAgentActionErrors, actionErrors]);

  // Use session storage state for pagination and sort
  const queryKeyPagination = JSON.stringify({
    pagination: {
      currentPage: page.index + 1,
      pageSize: page.size
    },
    sortField: sort.field,
    sortOrder: sort.direction
  });
  const queryKeyFilters = JSON.stringify({
    kuery,
    showAgentless,
    showInactive,
    showUpgradeable,
    search,
    selectedTags,
    selectedStatus
  });
  const {
    isInitialLoading,
    isFetching: isLoading,
    data,
    refetch
  } = (0, _reactQuery.useQuery)({
    queryKey: ['get-agents-list', queryKeyFilters, queryKeyPagination],
    keepPreviousData: true,
    // Keep previous data to avoid flashing when going through pages
    queryFn: async () => {
      try {
        var _managedAgentPolicies, _managedAgentPolicies2;
        const [agentsResponse, totalInactiveAgentsResponse, managedAgentPoliciesResponse, agentTagsResponse] = await Promise.all([(0, _hooks.sendGetAgentsForRq)({
          page: page.index + 1,
          perPage: page.size,
          kuery: kuery && kuery !== '' ? kuery : undefined,
          sortField: getSortFieldForAPI(sort.field),
          sortOrder: sort.direction,
          showAgentless,
          showInactive,
          showUpgradeable,
          getStatusSummary: true,
          withMetrics: true
        }), (0, _hooks.sendGetAgentStatus)({
          kuery: _services2.AgentStatusKueryHelper.buildKueryForInactiveAgents()
        }), (0, _hooks.sendGetAgentPolicies)({
          kuery: `${_constants.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE}.is_managed:true`,
          perPage: _constants.SO_SEARCH_LIMIT,
          full: false
        }), (0, _hooks.sendGetAgentTagsForRq)({
          showInactive
        })]);
        if (!totalInactiveAgentsResponse.data) {
          throw new Error('Invalid GET /agents_status response');
        }
        if (managedAgentPoliciesResponse.error) {
          throw new Error(managedAgentPoliciesResponse.error.message);
        }
        const statusSummary = agentsResponse.statusSummary;
        if (!statusSummary) {
          throw new Error('Invalid GET /agents response - no status summary');
        }
        // Fetch agent policies, use a local cache
        const policyIds = agentsResponse.items.map(agent => agent.policy_id);
        const policies = await fullAgentPolicyFecher.fetchPolicies(policyIds);
        const agentPoliciesIndexedById = policies.reduce((acc, agentPolicy) => {
          acc[agentPolicy.id] = agentPolicy;
          return acc;
        }, {});
        const agentsStatus = (0, _services.agentStatusesToSummary)(statusSummary);
        const newAllTags = [...agentTagsResponse.items];
        const agentsOnCurrentPage = agentsResponse.items;
        const nAgentsInTable = agentsResponse.total;
        const totalInactiveAgents = totalInactiveAgentsResponse.data.results.inactive || 0;
        const managedAgentPolicies = (_managedAgentPolicies = (_managedAgentPolicies2 = managedAgentPoliciesResponse.data) === null || _managedAgentPolicies2 === void 0 ? void 0 : _managedAgentPolicies2.items) !== null && _managedAgentPolicies !== void 0 ? _managedAgentPolicies : [];
        let totalManagedAgentIds = [];
        let managedAgentsOnCurrentPage = 0;
        if (managedAgentPolicies.length !== 0) {
          var _response$items;
          // Find all the agents that have managed policies
          // to the correct ids we need to build the kuery applying the same filters as the global ones
          const managedPoliciesKuery = (0, _get_kuery.getKuery)({
            search,
            selectedAgentPolicies: managedAgentPolicies.map(policy => policy.id),
            selectedTags,
            selectedStatus
          });
          const response = await (0, _hooks.sendGetAgentsForRq)({
            kuery: `${managedPoliciesKuery}`,
            perPage: _constants.SO_SEARCH_LIMIT,
            showInactive
          });
          const allManagedAgents = (_response$items = response === null || response === void 0 ? void 0 : response.items) !== null && _response$items !== void 0 ? _response$items : [];
          const allManagedAgentIds = allManagedAgents === null || allManagedAgents === void 0 ? void 0 : allManagedAgents.map(agent => agent.id);
          totalManagedAgentIds = allManagedAgentIds;
          managedAgentsOnCurrentPage = agentsResponse.items.map(agent => agent.id).filter(agentId => allManagedAgentIds.includes(agentId)).length;
        }
        return {
          agentPoliciesIndexedById,
          agentsStatus,
          agentsOnCurrentPage,
          nAgentsInTable,
          totalInactiveAgents,
          newAllTags,
          totalManagedAgentIds,
          managedAgentsOnCurrentPage,
          queryKeyFilters
        };
      } catch (error) {
        notifications.toasts.addError(error, {
          title: _i18n.i18n.translate('xpack.fleet.agentList.errorFetchingDataTitle', {
            defaultMessage: 'Error fetching agents'
          })
        });
        throw error;
      }
    },
    refetchInterval: REFRESH_INTERVAL_MS
  });
  const agentsStatus = data === null || data === void 0 ? void 0 : data.agentsStatus;
  const agentsOnCurrentPage = (0, _react.useMemo)(() => (data === null || data === void 0 ? void 0 : data.agentsOnCurrentPage) || [], [data]);
  const nAgentsInTable = (data === null || data === void 0 ? void 0 : data.nAgentsInTable) || 0;
  const totalInactiveAgents = (data === null || data === void 0 ? void 0 : data.totalInactiveAgents) || 0;
  const agentPoliciesIndexedById = (data === null || data === void 0 ? void 0 : data.agentPoliciesIndexedById) || {};
  const totalManagedAgentIds = (data === null || data === void 0 ? void 0 : data.totalManagedAgentIds) || [];
  const managedAgentsOnCurrentPage = (data === null || data === void 0 ? void 0 : data.managedAgentsOnCurrentPage) || 0;
  const newAllTags = (0, _react.useMemo)(() => (data === null || data === void 0 ? void 0 : data.newAllTags) || [], [data]);
  (0, _react.useEffect)(() => {
    if (newAllTags.length && !(0, _lodash.isEqual)(newAllTags, allTags)) {
      setAllTags(newAllTags);
    }
  }, [newAllTags, allTags]);
  const fetchData = (0, _react.useCallback)(async ({
    refreshTags = false
  } = {}) => {
    return refetch();
  }, [refetch]);
  const queryHasChanged = (0, _react.useMemo)(() => {
    return !(0, _lodash.isEqual)(queryKeyFilters, data === null || data === void 0 ? void 0 : data.queryKeyFilters);
  }, [queryKeyFilters, data === null || data === void 0 ? void 0 : data.queryKeyFilters]);
  const agentPoliciesRequest = (0, _hooks.useGetAgentPolicies)({
    page: 1,
    perPage: _constants.SO_SEARCH_LIMIT
  });
  const allAgentPolicies = (0, _react.useMemo)(() => {
    var _agentPoliciesRequest;
    return ((_agentPoliciesRequest = agentPoliciesRequest.data) === null || _agentPoliciesRequest === void 0 ? void 0 : _agentPoliciesRequest.items) || [];
  }, [agentPoliciesRequest.data]);
  return {
    allTags,
    setAllTags,
    agentsOnCurrentPage,
    agentsStatus,
    isLoading,
    isInitialLoading,
    nAgentsInTable,
    totalInactiveAgents,
    totalManagedAgentIds,
    managedAgentsOnCurrentPage,
    showUpgradeable,
    setShowUpgradeable,
    search,
    setSearch,
    selectedAgentPolicies,
    setSelectedAgentPolicies,
    sort,
    selectedStatus,
    setSelectedStatus,
    selectedTags,
    setSelectedTags,
    allAgentPolicies,
    agentPoliciesRequest,
    agentPoliciesIndexedById,
    page,
    pageSizeOptions,
    kuery,
    draftKuery,
    setDraftKuery,
    fetchData,
    queryHasChanged,
    latestAgentActionErrors,
    setLatestAgentActionErrors,
    isUsingFilter,
    clearFilters: sessionState.clearFilters,
    onTableChange: sessionState.onTableChange
  };
}