"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EsqlService = void 0;
var _esqlTypes = require("@kbn/esql-types");
var _utils = require("../lookup/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class EsqlService {
  constructor(options) {
    this.options = options;
  }

  /**
   * Get indices by their mode (lookup or time_series).
   * @param mode The mode to filter indices by.
   * @param remoteClusters Optional comma-separated list of remote clusters to include.
   * @returns A promise that resolves to the indices autocomplete result.
   */
  async getIndicesByIndexMode(mode, remoteClusters) {
    var _sources$indices, _sources$data_streams;
    const {
      client
    } = this.options;
    const indices = [];
    const sourcesToQuery = ['*'];
    const remoteClustersArray = [];
    if (remoteClusters) {
      remoteClustersArray.push(...remoteClusters.split(','));
      // attach a wildcard * for each remoteCluster
      const clustersArray = remoteClustersArray.map(cluster => `${cluster.trim()}:*`);
      sourcesToQuery.push(...clustersArray);
    }

    // It doesn't return hidden indices
    const sources = await client.indices.resolveIndex({
      name: sourcesToQuery,
      expand_wildcards: 'open',
      querystring: {
        mode
      }
    });
    (_sources$indices = sources.indices) === null || _sources$indices === void 0 ? void 0 : _sources$indices.forEach(index => {
      var _index$aliases;
      indices.push({
        name: index.name,
        mode,
        aliases: (_index$aliases = index.aliases) !== null && _index$aliases !== void 0 ? _index$aliases : []
      });
    });
    (_sources$data_streams = sources.data_streams) === null || _sources$data_streams === void 0 ? void 0 : _sources$data_streams.forEach(dataStream => {
      var _dataStream$aliases;
      indices.push({
        name: dataStream.name,
        mode,
        aliases: (_dataStream$aliases = dataStream.aliases) !== null && _dataStream$aliases !== void 0 ? _dataStream$aliases : []
      });
    });
    const crossClusterCommonIndices = remoteClusters ? (0, _utils.getListOfCCSIndices)(remoteClustersArray, indices) : indices;
    const result = {
      indices: crossClusterCommonIndices
    };
    return result;
  }

  /**
   * Get all indices, aliases, and data streams for ES|QL sources autocomplete.
   * @param scope The scope to retrieve indices for (local or all).
   * @returns A promise that resolves to an array of ESQL source results.
   */
  async getAllIndices(scope = 'local') {
    var _availableSources$ind, _availableSources$ali, _availableSources$dat;
    const {
      client
    } = this.options;

    // All means local + remote indices (queried with <cluster>:*)
    const namesToQuery = scope === 'local' ? ['*'] : ['*', '*:*'];

    // hidden and not, important for finding timeseries mode
    // mode is not returned for time_series datastreams, we need to find it from the indices
    // which are usually hidden
    const [allSources, availableSources] = await Promise.all([client.indices.resolveIndex({
      name: namesToQuery,
      expand_wildcards: 'all' // this returns hidden indices too
    }), client.indices.resolveIndex({
      name: namesToQuery,
      expand_wildcards: 'open'
    })]);
    const suggestedIndices = this.processSuggestedIndices((_availableSources$ind = availableSources.indices) !== null && _availableSources$ind !== void 0 ? _availableSources$ind : []);
    const suggestedAliases = this.processSuggestedAliases((_availableSources$ali = availableSources.aliases) !== null && _availableSources$ali !== void 0 ? _availableSources$ali : []);
    const suggestedDataStreams = this.processSuggestedDataStreams((_availableSources$dat = availableSources.data_streams) !== null && _availableSources$dat !== void 0 ? _availableSources$dat : [], allSources.indices);
    return [...suggestedIndices, ...suggestedAliases, ...suggestedDataStreams];
  }
  getIndexSourceType(mode) {
    const modeTypeMap = {
      time_series: _esqlTypes.SOURCES_TYPES.TIMESERIES,
      lookup: _esqlTypes.SOURCES_TYPES.LOOKUP
    };
    return modeTypeMap[mode !== null && mode !== void 0 ? mode : ''] || _esqlTypes.SOURCES_TYPES.INDEX;
  }
  processSuggestedIndices(indices) {
    var _indices$map;
    return (_indices$map = indices === null || indices === void 0 ? void 0 : indices.map(index => {
      // for remote clusters the format is cluster:indexName
      const [_, indexName] = index.name.split(':');
      return {
        name: index.name,
        type: this.getIndexSourceType(index.mode),
        // Extra hidden flag to flag system indices in the UI
        hidden: (indexName === null || indexName === void 0 ? void 0 : indexName.startsWith('.')) || index.name.startsWith('.')
      };
    })) !== null && _indices$map !== void 0 ? _indices$map : [];
  }
  processSuggestedAliases(aliases) {
    var _aliases$map;
    return (_aliases$map = aliases === null || aliases === void 0 ? void 0 : aliases.map(alias => {
      // for remote clusters the format is cluster:aliasName
      const [_, aliasName] = alias.name.split(':');
      return {
        name: alias.name,
        type: _esqlTypes.SOURCES_TYPES.ALIAS,
        // Extra hidden flag to flag system aliases in the UI
        hidden: (aliasName === null || aliasName === void 0 ? void 0 : aliasName.startsWith('.')) || alias.name.startsWith('.')
      };
    })) !== null && _aliases$map !== void 0 ? _aliases$map : [];
  }
  processSuggestedDataStreams(dataStreams, indices) {
    var _indices$map2, _dataStreams$map;
    const indexModeMap = new Map((_indices$map2 = indices === null || indices === void 0 ? void 0 : indices.map(idx => [idx.name, idx.mode])) !== null && _indices$map2 !== void 0 ? _indices$map2 : []);
    return (_dataStreams$map = dataStreams === null || dataStreams === void 0 ? void 0 : dataStreams.map(dataStream => {
      const backingIndices = dataStream.backing_indices || [];
      // Determine if any of the backing indices are time_series
      const isTimeSeries = backingIndices.some(indexName => indexModeMap.get(indexName) === 'time_series');
      return {
        name: dataStream.name,
        type: isTimeSeries ? _esqlTypes.SOURCES_TYPES.TIMESERIES : _esqlTypes.SOURCES_TYPES.DATA_STREAM,
        // Extra hidden flag to flag system data streams in the UI
        hidden: dataStream.name.startsWith('.')
      };
    })) !== null && _dataStreams$map !== void 0 ? _dataStreams$map : [];
  }

  /**
   * Get inference endpoints for a specific task type.
   * @param taskType The type of inference task to retrieve endpoints for.
   * @returns A promise that resolves to the inference endpoints autocomplete result.
   */
  async getInferenceEndpoints(taskType) {
    const {
      client
    } = this.options;
    const {
      endpoints
    } = await client.inference.get({
      inference_id: '_all',
      task_type: taskType
    });
    return {
      inferenceEndpoints: endpoints.map(endpoint => ({
        inference_id: endpoint.inference_id,
        task_type: endpoint.task_type
      }))
    };
  }
}
exports.EsqlService = EsqlService;