"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.inlineSuggest = inlineSuggest;
var _esqlAst = require("@kbn/esql-ast");
var _recommended_queries = require("@kbn/esql-ast/src/commands_registry/options/recommended_queries");
var _columns = require("../shared/columns");
var _resources_helpers = require("../shared/resources_helpers");
var _inline_suggestions_cache = require("./inline_suggestions_cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const FALLBACK_FROM_COMMAND = 'FROM *';
function getSourceFromQuery(esql) {
  var _sourceCommand$args;
  const queryExpression = _esqlAst.EsqlQuery.fromSrc(esql || '').ast;
  const sourceCommand = queryExpression.commands.find(({
    name
  }) => ['from', 'ts'].includes(name));
  const args = (_sourceCommand$args = sourceCommand === null || sourceCommand === void 0 ? void 0 : sourceCommand.args) !== null && _sourceCommand$args !== void 0 ? _sourceCommand$args : [];
  const indices = args.filter(arg => arg.sourceType === 'index');
  return indices === null || indices === void 0 ? void 0 : indices.map(index => index.name).join(',');
}

/**
 * Normalizes query text by removing comments and newlines
 */
function processQuery(query) {
  return query.replace(/\/\*[^*]*(?:\*(?!\/)[^*]*)*\*\//g, '') // Remove block comments (/* */)
  .replace(/\/\/.*$/gm, '') // Remove line comments (//)
  .replace(/[\n\r]/g, '') // Remove newlines
  .replace(/\s*\|\s*/g, ' | ') // Normalize pipe spacing
  .trim();
}

/**
 * Removes duplicate suggestions
 */
function removeDuplicates(suggestions) {
  const seenKeys = new Set();
  return suggestions.filter(item => {
    const key = processQuery(item.insertText).toLowerCase();
    if (seenKeys.has(key)) {
      return false;
    }
    seenKeys.add(key);
    return true;
  });
}

/**
 * Filters suggestions by prefix and ensures they're different from the current text
 */
function filterByPrefix(suggestions, prefix, fullText) {
  const normalizedFullText = processQuery(fullText).toLowerCase();
  return suggestions.filter(item => {
    const normalizedText = processQuery(item.insertText).toLowerCase();
    return normalizedText.startsWith(prefix) && normalizedText !== prefix && normalizedText !== normalizedFullText;
  });
}

/**
 * Trims the prefix from the insert text to show only the completion part
 */
function trimSuggestionPrefix(item, prefixLength) {
  return {
    ...item,
    insertText: processQuery(item.insertText).substring(prefixLength)
  };
}

/**
 * Fetches data sources and determines the FROM command
 */
async function getFromCommand(textBeforeCursor, callbacks) {
  // First, try to extract data source from the existing query
  const dataSource = getSourceFromQuery(textBeforeCursor);
  if (dataSource) {
    return `FROM ${dataSource}`;
  }
  const suggestedFromCommand = await (0, _resources_helpers.getFromCommandHelper)(callbacks);
  return suggestedFromCommand || FALLBACK_FROM_COMMAND;
}

/**
 * Fetches extension-based suggestions
 */
async function getExtensionSuggestions(fromCommand, range, callbacks) {
  var _callbacks$getEditorE;
  const editorExtensions = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getEditorE = callbacks.getEditorExtensions) === null || _callbacks$getEditorE === void 0 ? void 0 : _callbacks$getEditorE.call(callbacks, fromCommand).then(result => result !== null && result !== void 0 ? result : {
    recommendedQueries: []
  }));
  return ((editorExtensions === null || editorExtensions === void 0 ? void 0 : editorExtensions.recommendedQueries) || []).map(query => ({
    insertText: query.query,
    range
  }));
}

/**
 * Fetches history-based suggestions
 */
async function getHistorySuggestions(range, callbacks) {
  var _callbacks$getHistory;
  const historyStarredItems = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getHistory = callbacks.getHistoryStarredItems) === null || _callbacks$getHistory === void 0 ? void 0 : _callbacks$getHistory.call(callbacks).then(result => result !== null && result !== void 0 ? result : []));
  return (historyStarredItems || []).map(item => ({
    insertText: item,
    range
  }));
}

/**
 * Gets template-based suggestions with caching
 */
function getCachedRecommendedTemplateSuggestions(fromCommand, timeField, categorizationField, range) {
  const cacheKey = `${fromCommand}:${timeField}:${categorizationField}`;
  const cached = (0, _inline_suggestions_cache.fromCache)(cacheKey);
  if (cached) {
    // Update range for cached results since range can change
    return cached.map(item => ({
      ...item,
      range
    }));
  }
  const suggestions = (0, _recommended_queries.getRecommendedQueriesTemplates)({
    fromCommand,
    timeField,
    categorizationField
  }).map(query => ({
    insertText: query.queryString,
    range
  }));

  // Cache the templates
  (0, _inline_suggestions_cache.setToCache)(cacheKey, suggestions);
  return suggestions;
}

/**
 * Fetches all suggestion sources
 */
async function fetchAllSuggestions(fromCommand, timeField, categorizationField, range, callbacks) {
  const [extensionSuggestions, historySuggestions] = await Promise.all([getExtensionSuggestions(fromCommand, range, callbacks), getHistorySuggestions(range, callbacks)]);
  const templateSuggestions = getCachedRecommendedTemplateSuggestions(fromCommand, timeField, categorizationField, range);
  return [...extensionSuggestions, ...templateSuggestions, ...historySuggestions];
}

/**
 * Gets field information with caching
 */
async function getCachedTimeAndCategorizationFields(fromCommand, callbacks) {
  const cacheKey = `${fromCommand}`;
  const cached = (0, _inline_suggestions_cache.fromCache)(cacheKey);
  if (cached) {
    return cached;
  }
  const {
    getColumnsByType
  } = (0, _columns.getColumnsByTypeRetriever)(_esqlAst.EsqlQuery.fromSrc(fromCommand).ast, fromCommand, callbacks);
  const fieldInfo = await (0, _recommended_queries.getTimeAndCategorizationFields)(getColumnsByType);

  // Cache the field information
  (0, _inline_suggestions_cache.setToCache)(cacheKey, fieldInfo);
  return fieldInfo;
}
async function inlineSuggest(fullText, textBeforeCursor, range, callbacks) {
  try {
    const trimmedText = processQuery(textBeforeCursor).toLowerCase();
    const trimmedFullText = processQuery(fullText).toLowerCase();
    if (trimmedText !== trimmedFullText) {
      // Don't show suggestions if cursor is not at the end of the query
      return {
        items: []
      };
    }

    // Important for recommended queries
    const fromCommand = await getFromCommand(trimmedText, callbacks);
    const {
      timeField,
      categorizationField
    } = await getCachedTimeAndCategorizationFields(fromCommand, callbacks);

    // Fetch all suggestions
    const allSuggestions = await fetchAllSuggestions(fromCommand, timeField, categorizationField, range, callbacks);

    // Process suggestions: remove duplicates, filter by prefix, and trim prefix
    const uniqueSuggestions = removeDuplicates(allSuggestions);
    const filteredSuggestions = filterByPrefix(uniqueSuggestions, trimmedText, fullText);
    const finalSuggestions = filteredSuggestions.map(item => trimSuggestionPrefix(item, trimmedText.length));
    return {
      items: finalSuggestions
    };
  } catch (error) {
    return {
      items: []
    };
  }
}