"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFunctionSignatureHover = getFunctionSignatureHover;
var _esqlAst = require("@kbn/esql-ast");
var _utils = require("@kbn/esql-ast/src/definitions/utils");
var _hover_cache = require("./hover_cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function getFunctionSignatureHover(fnNode, getColumnMap) {
  // Getting the columns map is not expensive, it's already cached.
  const columnsMap = await getColumnMap();

  // Use function name and argument types as cache key, fnName:argType1,argType2
  const cacheKey = getFunctionCachekey(fnNode, columnsMap);
  const cached = (0, _hover_cache.fromCache)(cacheKey);
  if (cached) {
    return cached;
  }
  const fnDefinition = (0, _utils.getFunctionDefinition)(fnNode.name);
  if (fnDefinition) {
    const formattedSignature = (0, _utils.getFormattedFunctionSignature)(fnDefinition, fnNode, columnsMap);
    const result = [{
      value: `\`\`\`none
${formattedSignature}
\`\`\``
    }, {
      value: fnDefinition.description
    }];
    (0, _hover_cache.setToCache)(cacheKey, result);
    return result;
  } else {
    (0, _hover_cache.setToCache)(cacheKey, []);
    return [];
  }
}

/**
 * Returns a cache key for the function signature hover based on function name and argument types.
 * fnName:argType1,argType2
 *
 * @param fnNode
 * @param columnsMap
 * @returns
 */
const getFunctionCachekey = (fnNode, columnsMap) => {
  const argTypes = [];
  _esqlAst.Walker.walk(fnNode, {
    visitColumn: columnNode => {
      var _columnsMap$get;
      const columnType = ((_columnsMap$get = columnsMap.get(columnNode.name)) === null || _columnsMap$get === void 0 ? void 0 : _columnsMap$get.type) || 'unknown';
      argTypes.push(columnType);
    }
  });
  return `${fnNode.name}:${argTypes.join(',')}`;
};