"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateReactRouterProps = void 0;
var _http = require("../http");
var _kibana = require("../kibana");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Generates the `href` and `onClick` props for React-Router-friendly links
 *
 * Based off of EUI's recommendations for handling React Router:
 * https://github.com/elastic/eui/blob/main/wiki/consuming-eui/react-router.md#react-router-51
 *
 * but separated out from EuiLink portion as we use this for multiple EUI components
 */

const generateReactRouterProps = ({
  to,
  onClick,
  shouldNotCreateHref = false,
  shouldNotPrepend = false
}) => {
  const {
    navigateToUrl,
    history
  } = _kibana.KibanaLogic.values;
  const {
    http
  } = _http.HttpLogic.values;

  // Generate the correct link href (with basename etc. accounted for)
  const href = (0, _.createHref)(to, {
    history,
    http
  }, {
    shouldNotCreateHref,
    shouldNotPrepend
  });
  const reactRouterLinkClick = event => {
    if (onClick) onClick(); // Run any passed click events (e.g. telemetry)
    if ((0, _.letBrowserHandleEvent)(event)) return; // Return early if the link behavior shouldn't be handled by React Router

    // Prevent regular link behavior, which causes a browser refresh.
    event.preventDefault();

    // Perform SPA navigation.
    navigateToUrl(to, {
      shouldNotCreateHref,
      shouldNotPrepend
    });
  };
  return {
    href,
    onClick: reactRouterLinkClick
  };
};
exports.generateReactRouterProps = generateReactRouterProps;