"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateConversationRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _schemas = require("@kbn/elastic-assistant-common/impl/schemas");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _audit_events = require("../../ai_assistant_data_clients/knowledge_base/audit_events");
var _event_based_telemetry = require("../../lib/telemetry/event_based_telemetry");
var _utils = require("../utils");
var _helpers = require("../helpers");
var _audit_events2 = require("../../ai_assistant_data_clients/conversations/audit_events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateConversationRoute = router => {
  router.versioned.put({
    access: 'public',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_CONVERSATIONS_URL_BY_ID,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_schemas.ConversationUpdateProps),
        params: (0, _common.buildRouteValidationWithZod)(_schemas.UpdateConversationRequestParams)
      }
    }
  }, async (context, request, response) => {
    const assistantResponse = (0, _utils.buildResponse)(response);
    const {
      id
    } = request.params;
    let telemetry;
    let auditLogger;
    let authenticatedUser;
    try {
      var _authenticatedUser, _authenticatedUser2;
      const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
      telemetry = ctx.elasticAssistant.telemetry;
      auditLogger = ctx.elasticAssistant.auditLogger;
      // Perform license and authenticated user checks
      const checkResponse = await (0, _helpers.performChecks)({
        context: ctx,
        request,
        response
      });
      if (!checkResponse.isSuccess) {
        return checkResponse.response;
      }
      authenticatedUser = checkResponse.currentUser;
      const dataClient = await ctx.elasticAssistant.getAIAssistantConversationsDataClient();
      const existingConversation = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.getConversation({
        id,
        authenticatedUser
      }));
      if (existingConversation == null) {
        return assistantResponse.error({
          body: `conversation id: "${id}" not found`,
          statusCode: 404
        });
      }
      if (!(0, _elasticAssistantCommon.getIsConversationOwner)(existingConversation, {
        name: (_authenticatedUser = authenticatedUser) === null || _authenticatedUser === void 0 ? void 0 : _authenticatedUser.username,
        id: (_authenticatedUser2 = authenticatedUser) === null || _authenticatedUser2 === void 0 ? void 0 : _authenticatedUser2.profile_uid
      })) {
        return assistantResponse.error({
          body: `conversation id: "${id}". Updating a conversation is only allowed for the owner of the conversation.`,
          statusCode: 403
        });
      }
      const conversation = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.updateConversation({
        conversationUpdateProps: {
          ...request.body,
          id
        }
      }));
      if (request.body.users) {
        var _auditLogger;
        const conversationSharedState = (0, _elasticAssistantCommon.getConversationSharedState)({
          users: request.body.users,
          id
        });
        telemetry.reportEvent(_event_based_telemetry.CONVERSATION_SHARED_SUCCESS_EVENT.eventType, {
          sharing: conversationSharedState,
          ...(conversationSharedState === _elasticAssistantCommon.ConversationSharedState.RESTRICTED ?
          // if restricted, track number of additional users added (minus the owner)
          {
            total: request.body.users.length - 1
          } : {})
        });
        (_auditLogger = auditLogger) === null || _auditLogger === void 0 ? void 0 : _auditLogger.log((0, _audit_events2.conversationAuditEvent)({
          action: (0, _audit_events2.getAuditAction)(conversationSharedState),
          id: conversation === null || conversation === void 0 ? void 0 : conversation.id,
          title: conversation === null || conversation === void 0 ? void 0 : conversation.title,
          users: request.body.users.filter(u => {
            var _authenticatedUser3, _authenticatedUser4;
            return u.id !== ((_authenticatedUser3 = authenticatedUser) === null || _authenticatedUser3 === void 0 ? void 0 : _authenticatedUser3.profile_uid) && u.name !== ((_authenticatedUser4 = authenticatedUser) === null || _authenticatedUser4 === void 0 ? void 0 : _authenticatedUser4.username);
          }),
          outcome: _audit_events.AUDIT_OUTCOME.SUCCESS
        }));
      }
      if (conversation == null) {
        return assistantResponse.error({
          body: `conversation id: "${id}" was not updated`,
          statusCode: 400
        });
      }
      return response.ok({
        body: conversation
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      if (request.body.users) {
        var _telemetry, _auditLogger2;
        const conversationSharedState = (0, _elasticAssistantCommon.getConversationSharedState)({
          users: request.body.users,
          id
        });
        (_telemetry = telemetry) === null || _telemetry === void 0 ? void 0 : _telemetry.reportEvent(_event_based_telemetry.CONVERSATION_SHARED_ERROR_EVENT.eventType, {
          sharing: conversationSharedState,
          errorMessage: error.message
        });
        (_auditLogger2 = auditLogger) === null || _auditLogger2 === void 0 ? void 0 : _auditLogger2.log((0, _audit_events2.conversationAuditEvent)({
          action: conversationSharedState === _elasticAssistantCommon.ConversationSharedState.PRIVATE ? _audit_events2.ConversationAuditAction.PRIVATE : _audit_events2.ConversationAuditAction.SHARED,
          id: request.body.id,
          users: request.body.users.filter(u => {
            var _authenticatedUser5, _authenticatedUser6;
            return u.id !== ((_authenticatedUser5 = authenticatedUser) === null || _authenticatedUser5 === void 0 ? void 0 : _authenticatedUser5.profile_uid) && u.name !== ((_authenticatedUser6 = authenticatedUser) === null || _authenticatedUser6 === void 0 ? void 0 : _authenticatedUser6.username);
          }),
          outcome: _audit_events.AUDIT_OUTCOME.FAILURE,
          error: err
        }));
      }
      return assistantResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.updateConversationRoute = updateConversationRoute;