"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readConversationRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _schemas = require("@kbn/elastic-assistant-common/impl/schemas");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _event_based_telemetry = require("../../lib/telemetry/event_based_telemetry");
var _utils = require("../utils");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const readConversationRoute = router => {
  router.versioned.get({
    access: 'public',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_CONVERSATIONS_URL_BY_ID,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        params: (0, _common.buildRouteValidationWithZod)(_schemas.ReadConversationRequestParams)
      }
    }
  }, async (context, request, response) => {
    const assistantResponse = (0, _utils.buildResponse)(response);
    const {
      id
    } = request.params;
    try {
      var _checkResponse$curren, _checkResponse$curren2;
      const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
      const checkResponse = await (0, _helpers.performChecks)({
        context: ctx,
        request,
        response
      });
      if (!checkResponse.isSuccess) {
        return checkResponse.response;
      }
      const authenticatedUser = checkResponse.currentUser;
      const dataClient = await ctx.elasticAssistant.getAIAssistantConversationsDataClient();

      // First check if the conversation exists at all
      const conversationExists = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.conversationExists({
        id
      }));
      if (!conversationExists) {
        return assistantResponse.error({
          body: `conversation id: "${id}" not found`,
          statusCode: 404
        });
      }

      // Then check if the user has access to the conversation
      const conversation = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.getConversation({
        id,
        authenticatedUser
      }));
      if (conversation == null) {
        return assistantResponse.error({
          body: `Access denied to conversation id: "${id}"`,
          statusCode: 403
        });
      }
      const isConversationOwner = (0, _elasticAssistantCommon.getIsConversationOwner)(conversation, {
        name: (_checkResponse$curren = checkResponse.currentUser) === null || _checkResponse$curren === void 0 ? void 0 : _checkResponse$curren.username,
        id: (_checkResponse$curren2 = checkResponse.currentUser) === null || _checkResponse$curren2 === void 0 ? void 0 : _checkResponse$curren2.profile_uid
      });
      if (!isConversationOwner) {
        const telemetry = ctx.elasticAssistant.telemetry;
        telemetry.reportEvent(_event_based_telemetry.SHARED_CONVERSATION_ACCESSED_EVENT.eventType, {
          sharing: (0, _elasticAssistantCommon.getConversationSharedState)({
            users: conversation.users,
            id: conversation.id
          })
        });
      }
      return response.ok({
        body: conversation
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return assistantResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.readConversationRoute = readConversationRoute;