"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getKnowledgeBaseStatusRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _build_response = require("../../lib/build_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get the status of the Knowledge Base index, pipeline, and resources (collection of documents)
 *
 * @param router IRouter for registering routes
 */
const getKnowledgeBaseStatusRoute = router => {
  router.versioned.get({
    access: 'public',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_KNOWLEDGE_BASE_URL,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        params: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ReadKnowledgeBaseRequestParams)
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const assistantContext = ctx.elasticAssistant;
    const logger = ctx.elasticAssistant.logger;
    try {
      const kbDataClient = await assistantContext.getAIAssistantKnowledgeBaseDataClient();
      if (!kbDataClient) {
        return response.custom({
          body: {
            success: false
          },
          statusCode: 500
        });
      }
      const setupAvailable = await kbDataClient.isSetupAvailable();
      const isInferenceEndpointExists = await kbDataClient.isInferenceEndpointExists();
      const securityLabsExists = await kbDataClient.isSecurityLabsDocsLoaded();
      const loadedSecurityLabsDocsCount = await kbDataClient.getLoadedSecurityLabsDocsCount();
      const userDataExists = await kbDataClient.isUserDataExists();
      const productDocumentationStatus = await kbDataClient.getProductDocumentationStatus();
      const defendInsightsExists = await kbDataClient.isDefendInsightsDocsLoaded();
      return response.ok({
        body: {
          elser_exists: isInferenceEndpointExists,
          is_setup_in_progress: kbDataClient.isSetupInProgress,
          is_setup_available: setupAvailable,
          security_labs_exists: securityLabsExists,
          defend_insights_exists: defendInsightsExists,
          // If user data exists, we should have at least one document in the Security Labs index
          user_data_exists: userDataExists || !!loadedSecurityLabsDocsCount,
          product_documentation_status: productDocumentationStatus
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.getKnowledgeBaseStatusRoute = getKnowledgeBaseStatusRoute;