"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildEsqlFetchSubscribe = void 0;
var _esQuery = require("@kbn/es-query");
var _esqlUtils = require("@kbn/esql-utils");
var _lodash = require("lodash");
var _types = require("../../../types");
var _redux = require("../redux");
var _get_valid_view_mode = require("../../utils/get_valid_view_mode");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const ESQL_MAX_NUM_OF_COLUMNS = 50;

/*
 * Takes care of ES|QL state transformations when a new result is returned
 * If necessary this is setting displayed columns and selected data view
 */
const buildEsqlFetchSubscribe = ({
  internalState,
  appStateContainer,
  dataSubjects,
  injectCurrentTab
}) => {
  let prevEsqlData = {
    initialFetch: true,
    query: '',
    allColumns: [],
    defaultColumns: []
  };
  const cleanupEsql = () => {
    if (!prevEsqlData.query) {
      return;
    }

    // cleanup when it's not an ES|QL query
    prevEsqlData = {
      initialFetch: true,
      query: '',
      allColumns: [],
      defaultColumns: []
    };
  };
  const esqlFetchSubscribe = async next => {
    var _next$result;
    const {
      query: nextQuery
    } = next;
    if (!nextQuery) {
      return;
    }
    if (!(0, _esQuery.isOfAggregateQueryType)(nextQuery)) {
      // cleanup for a "regular" query
      cleanupEsql();
      return;
    }

    // We need to reset the default profile state on index pattern changes
    // when loading starts to ensure the correct pre fetch state is available
    // before data fetching is triggered
    if (next.fetchStatus === _types.FetchStatus.LOADING) {
      // We have to grab the current query from appState
      // here since nextQuery has not been updated yet
      const appStateQuery = appStateContainer.get().query;
      if ((0, _esQuery.isOfAggregateQueryType)(appStateQuery)) {
        if (prevEsqlData.initialFetch) {
          prevEsqlData.query = appStateQuery.esql;
        }
        const indexPatternChanged = (0, _esqlUtils.getIndexPatternFromESQLQuery)(appStateQuery.esql) !== (0, _esqlUtils.getIndexPatternFromESQLQuery)(prevEsqlData.query);

        // Reset all default profile state when index pattern changes
        if (indexPatternChanged) {
          internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setResetDefaultProfileState)({
            resetDefaultProfileState: {
              columns: true,
              rowHeight: true,
              breakdownField: true,
              hideChart: true
            }
          }));
        }
      }
      return;
    }
    if (next.fetchStatus === _types.FetchStatus.ERROR) {
      // An error occurred, but it's still considered an initial fetch
      prevEsqlData.initialFetch = false;
      return;
    }
    if (next.fetchStatus !== _types.FetchStatus.PARTIAL) {
      return;
    }
    let nextAllColumns = prevEsqlData.allColumns;
    let nextDefaultColumns = prevEsqlData.defaultColumns;
    if ((_next$result = next.result) !== null && _next$result !== void 0 && _next$result.length) {
      nextAllColumns = Object.keys(next.result[0].raw);
      if ((0, _esqlUtils.hasTransformationalCommand)(nextQuery.esql)) {
        nextDefaultColumns = nextAllColumns.slice(0, ESQL_MAX_NUM_OF_COLUMNS);
      } else {
        nextDefaultColumns = [];
      }
    }
    if (prevEsqlData.initialFetch) {
      prevEsqlData.initialFetch = false;
      prevEsqlData.query = nextQuery.esql;
      prevEsqlData.allColumns = nextAllColumns;
      prevEsqlData.defaultColumns = nextDefaultColumns;
    }
    const indexPatternChanged = (0, _esqlUtils.getIndexPatternFromESQLQuery)(nextQuery.esql) !== (0, _esqlUtils.getIndexPatternFromESQLQuery)(prevEsqlData.query);
    const allColumnsChanged = !(0, _lodash.isEqual)(nextAllColumns, prevEsqlData.allColumns);
    const changeDefaultColumns = indexPatternChanged || !(0, _lodash.isEqual)(nextDefaultColumns, prevEsqlData.defaultColumns);
    const {
      viewMode
    } = appStateContainer.get();
    const changeViewMode = viewMode !== (0, _get_valid_view_mode.getValidViewMode)({
      viewMode,
      isEsqlMode: true
    });

    // If the index pattern hasn't changed, but the available columns have changed
    // due to transformational commands, reset the associated default profile state
    if (!indexPatternChanged && allColumnsChanged) {
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setResetDefaultProfileState)({
        resetDefaultProfileState: {
          columns: true,
          rowHeight: false,
          breakdownField: false,
          hideChart: false
        }
      }));
    }
    prevEsqlData.allColumns = nextAllColumns;
    if (indexPatternChanged || changeDefaultColumns || changeViewMode) {
      prevEsqlData.query = nextQuery.esql;
      prevEsqlData.defaultColumns = nextDefaultColumns;

      // just change URL state if necessary
      if (changeDefaultColumns || changeViewMode) {
        const nextState = {
          ...(changeDefaultColumns && {
            columns: nextDefaultColumns
          }),
          ...(changeViewMode && {
            viewMode: undefined
          })
        };
        await appStateContainer.replaceUrlState(nextState);
      }
    }
    dataSubjects.documents$.next({
      ...next,
      fetchStatus: _types.FetchStatus.COMPLETE
    });
  };
  return {
    esqlFetchSubscribe,
    cleanupEsql
  };
};
exports.buildEsqlFetchSubscribe = buildEsqlFetchSubscribe;