"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCurrentUrlState = getCurrentUrlState;
exports.getDiscoverAppStateContainer = void 0;
exports.getInitialState = getInitialState;
exports.isEqualFilters = isEqualFilters;
exports.isEqualState = isEqualState;
exports.setState = setState;
exports.useAppStateSelector = void 0;
var _esQuery = require("@kbn/es-query");
var _public = require("@kbn/kibana-utils-plugin/public");
var _lodash = require("lodash");
var _public2 = require("@kbn/data-plugin/public");
var _rxjs = require("rxjs");
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _add_log = require("../../../utils/add_log");
var _cleanup_url_state = require("./utils/cleanup_url_state");
var _get_state_defaults = require("./utils/get_state_defaults");
var _state_helpers = require("../../../utils/state_helpers");
var _data_sources = require("../../../../common/data_sources");
var _redux = require("./redux");
var _common = require("../../../../common");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const useAppStateSelector = selector => (0, _redux.useCurrentTabSelector)(tab => selector(tab.appState), _fastDeepEqual.default);

/**
 * This is the app state container for Discover main, it's responsible for syncing state with the URL
 * @param stateStorage
 * @param savedSearch
 * @param services
 */
exports.useAppStateSelector = useAppStateSelector;
const getDiscoverAppStateContainer = ({
  tabId,
  stateStorage,
  internalState,
  savedSearchContainer,
  services,
  injectCurrentTab
}) => {
  const getAppState = state => {
    return (0, _redux.selectTab)(state, tabId).appState;
  };
  const appStateContainer = {
    get: () => getAppState(internalState.getState()),
    set: appState => {
      if (!appState) {
        return;
      }
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setAppState)({
        appState
      }));
    },
    state$: (0, _rxjs.from)(internalState).pipe((0, _rxjs.map)(getAppState), (0, _rxjs.distinctUntilChanged)(_lodash.isEqual))
  };
  const getAppStateFromSavedSearch = newSavedSearch => {
    return getInitialState({
      initialUrlState: undefined,
      savedSearch: newSavedSearch,
      services
    });
  };
  const replaceUrlState = async (newPartial = {}, merge = true) => {
    (0, _add_log.addLog)('[appState] replaceUrlState', {
      newPartial,
      merge
    });
    const state = merge ? {
      ...appStateContainer.get(),
      ...newPartial
    } : newPartial;
    if (internalState.getState().tabs.unsafeCurrentId === tabId) {
      await stateStorage.set(_common.APP_STATE_URL_KEY, state, {
        replace: true
      });
    } else {
      appStateContainer.set(state);
    }
  };
  const getGlobalState = state => {
    const tabState = (0, _redux.selectTab)(state, tabId);
    const {
      timeRange: time,
      refreshInterval,
      filters
    } = tabState.globalState;
    return {
      time,
      refreshInterval,
      filters
    };
  };
  const globalStateContainer = {
    get: () => getGlobalState(internalState.getState()),
    set: state => {
      if (!state) {
        return;
      }
      const {
        time: timeRange,
        refreshInterval,
        filters
      } = state;
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setGlobalState)({
        globalState: {
          timeRange,
          refreshInterval,
          filters
        }
      }));
    },
    state$: (0, _rxjs.from)(internalState).pipe((0, _rxjs.map)(getGlobalState), (0, _rxjs.distinctUntilChanged)(_lodash.isEqual))
  };
  const updateUrlWithCurrentState = async () => {
    await Promise.all([stateStorage.set(_constants.GLOBAL_STATE_URL_KEY, globalStateContainer.get(), {
      replace: true
    }), replaceUrlState({})]);
  };
  const initAndSync = () => {
    const currentSavedSearch = savedSearchContainer.getState();
    (0, _add_log.addLog)('[appState] initialize state and sync with URL', currentSavedSearch);

    // Set the default profile state only if not loading a saved search,
    // to avoid overwriting saved search state
    if (!currentSavedSearch.id) {
      const {
        breakdownField,
        columns,
        rowHeight,
        hideChart
      } = getCurrentUrlState(stateStorage, services);

      // Only set default state which is not already set in the URL
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setResetDefaultProfileState)({
        resetDefaultProfileState: {
          columns: columns === undefined,
          rowHeight: rowHeight === undefined,
          breakdownField: breakdownField === undefined,
          hideChart: hideChart === undefined
        }
      }));
    }
    const {
      data
    } = services;
    const savedSearchDataView = currentSavedSearch.searchSource.getField('index');
    const appState = appStateContainer.get();
    const setDataViewFromSavedSearch = !appState.dataSource || (0, _data_sources.isDataSourceType)(appState.dataSource, _data_sources.DataSourceType.DataView) && appState.dataSource.dataViewId !== (savedSearchDataView === null || savedSearchDataView === void 0 ? void 0 : savedSearchDataView.id);
    if (setDataViewFromSavedSearch) {
      // used data view is different from the given by url/state which is invalid
      setState(appStateContainer, {
        dataSource: savedSearchDataView !== null && savedSearchDataView !== void 0 && savedSearchDataView.id ? (0, _data_sources.createDataViewDataSource)({
          dataViewId: savedSearchDataView.id
        }) : undefined
      });
    }

    // syncs `_a` portion of url with query services
    const stopSyncingQueryAppStateWithStateContainer = (0, _public2.connectToQueryState)(data.query, appStateContainer, {
      filters: _esQuery.FilterStateStore.APP_STATE,
      query: true
    });
    const {
      start: startSyncingAppStateWithUrl,
      stop: stopSyncingAppStateWithUrl
    } = (0, _public.syncState)({
      storageKey: _common.APP_STATE_URL_KEY,
      stateContainer: appStateContainer,
      stateStorage
    });

    // syncs `_g` portion of url with query services
    const stopSyncingQueryGlobalStateWithStateContainer = (0, _public2.connectToQueryState)(data.query, globalStateContainer, {
      refreshInterval: true,
      time: true,
      filters: _esQuery.FilterStateStore.GLOBAL_STATE
    });
    const {
      start: startSyncingGlobalStateWithUrl,
      stop: stopSyncingGlobalStateWithUrl
    } = (0, _public.syncState)({
      storageKey: _constants.GLOBAL_STATE_URL_KEY,
      stateContainer: globalStateContainer,
      stateStorage
    });

    // current state needs to be pushed to url
    updateUrlWithCurrentState().then(() => {
      startSyncingAppStateWithUrl();
      startSyncingGlobalStateWithUrl();
    });
    return () => {
      stopSyncingQueryAppStateWithStateContainer();
      stopSyncingQueryGlobalStateWithStateContainer();
      stopSyncingAppStateWithUrl();
      stopSyncingGlobalStateWithUrl();
    };
  };
  const update = (newPartial, replace = false) => {
    (0, _add_log.addLog)('[appState] update', {
      newPartial,
      replace
    });
    if (replace) {
      return replaceUrlState(newPartial);
    } else {
      setState(appStateContainer, newPartial);
    }
  };
  return {
    ...appStateContainer,
    initAndSync,
    updateUrlWithCurrentState,
    replaceUrlState,
    update,
    getAppStateFromSavedSearch
  };
};
exports.getDiscoverAppStateContainer = getDiscoverAppStateContainer;
function getCurrentUrlState(stateStorage, services) {
  var _cleanupUrlState, _stateStorage$get;
  return (_cleanupUrlState = (0, _cleanup_url_state.cleanupUrlState)((_stateStorage$get = stateStorage.get(_common.APP_STATE_URL_KEY)) !== null && _stateStorage$get !== void 0 ? _stateStorage$get : {}, services.uiSettings)) !== null && _cleanupUrlState !== void 0 ? _cleanupUrlState : {};
}
function getInitialState({
  initialUrlState,
  savedSearch,
  overrideDataView,
  services
}) {
  const defaultAppState = (0, _get_state_defaults.getStateDefaults)({
    savedSearch,
    overrideDataView,
    services
  });
  const mergedState = {
    ...defaultAppState,
    ...initialUrlState
  };

  // https://github.com/elastic/kibana/issues/122555
  if (typeof mergedState.hideChart !== 'boolean') {
    mergedState.hideChart = undefined;
  }

  // Don't allow URL state to overwrite the data source if there's an ES|QL query
  if ((0, _esQuery.isOfAggregateQueryType)(mergedState.query) && !(0, _data_sources.isEsqlSource)(mergedState.dataSource)) {
    mergedState.dataSource = (0, _data_sources.createEsqlDataSource)();
  }
  return (0, _state_helpers.handleSourceColumnState)(mergedState, services.uiSettings);
}

/**
 * Helper function to merge a given new state with the existing state and to set the given state
 * container
 */
function setState(stateContainer, newState) {
  (0, _add_log.addLog)('[appstate] setState', {
    newState
  });
  const oldState = stateContainer.get();
  const mergedState = {
    ...oldState,
    ...newState
  };
  if (!isEqualState(oldState, mergedState)) {
    stateContainer.set(mergedState);
  }
}

/**
 * Helper function to compare 2 different filter states
 */
function isEqualFilters(filtersA, filtersB, comparatorOptions = _esQuery.COMPARE_ALL_OPTIONS) {
  if (!filtersA && !filtersB) {
    return true;
  } else if (!filtersA || !filtersB) {
    return false;
  }
  return (0, _esQuery.compareFilters)(filtersA, filtersB, comparatorOptions);
}

/**
 * Helper function to compare 2 different state, is needed since comparing filters
 * works differently
 */
function isEqualState(stateA, stateB, exclude = []) {
  if (!stateA && !stateB) {
    return true;
  } else if (!stateA || !stateB) {
    return false;
  }
  const {
    filters: stateAFilters = [],
    ...stateAPartial
  } = (0, _lodash.omit)(stateA, exclude);
  const {
    filters: stateBFilters = [],
    ...stateBPartial
  } = (0, _lodash.omit)(stateB, exclude);
  return (0, _lodash.isEqual)(stateAPartial, stateBPartial) && isEqualFilters(stateAFilters, stateBFilters);
}