"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeUnsavedChangesManager = initializeUnsavedChangesManager;
var _presentationContainers = require("@kbn/presentation-containers");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _rxjs = require("rxjs");
var _dashboard_backup_service = require("../services/dashboard_backup_service");
var _control_group_manager = require("./control_group_manager");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEBOUNCE_TIME = 100;
function initializeUnsavedChangesManager({
  layoutManager,
  savedObjectId$,
  lastSavedState,
  settingsManager,
  viewMode$,
  storeUnsavedChanges,
  controlGroupManager,
  getReferences,
  unifiedSearchManager
}) {
  const hasUnsavedChanges$ = new _rxjs.BehaviorSubject(false);
  const lastSavedState$ = new _rxjs.BehaviorSubject(lastSavedState);
  const hasChildrenUnsavedChanges$ = (0, _presentationContainers.childrenUnsavedChanges$)(layoutManager.api.children$).pipe((0, _rxjs.tap)(childrenWithChanges => {
    // propagate the latest serialized state back to the layout manager.
    for (const {
      uuid,
      hasUnsavedChanges
    } of childrenWithChanges) {
      const childApi = layoutManager.api.children$.value[uuid];
      if (!hasUnsavedChanges || !childApi || !(0, _presentationPublishing.apiHasSerializableState)(childApi)) continue;
      layoutManager.internalApi.setChildState(uuid, childApi.serializeState());
    }
  }), (0, _rxjs.map)(childrenWithChanges => {
    return childrenWithChanges.some(({
      hasUnsavedChanges
    }) => hasUnsavedChanges);
  }));
  const dashboardStateChanges$ = (0, _rxjs.combineLatest)([settingsManager.internalApi.startComparing$(lastSavedState$), unifiedSearchManager.internalApi.startComparing$(lastSavedState$), layoutManager.internalApi.startComparing$(lastSavedState$)]).pipe((0, _rxjs.map)(([settings, unifiedSearch, panels]) => {
    return {
      ...settings,
      ...unifiedSearch,
      ...panels
    };
  }));
  const unsavedChangesSubscription = (0, _rxjs.combineLatest)([viewMode$, dashboardStateChanges$, hasChildrenUnsavedChanges$, controlGroupManager.api.controlGroupApi$.pipe((0, _rxjs.skipWhile)(controlGroupApi => !controlGroupApi), (0, _rxjs.switchMap)(controlGroupApi => {
    return controlGroupApi.hasUnsavedChanges$;
  }))]).pipe((0, _rxjs.debounceTime)(DEBOUNCE_TIME)).subscribe(([viewMode, dashboardChanges, hasChildrenUnsavedChanges, hasControlGroupChanges]) => {
    const hasDashboardChanges = Object.keys(dashboardChanges !== null && dashboardChanges !== void 0 ? dashboardChanges : {}).length > 0;
    const hasLayoutChanges = dashboardChanges.panels;
    const hasUnsavedChanges = hasDashboardChanges || hasChildrenUnsavedChanges || hasControlGroupChanges;
    if (hasUnsavedChanges !== hasUnsavedChanges$.value) {
      hasUnsavedChanges$.next(hasUnsavedChanges);
    }
    if (storeUnsavedChanges) {
      const {
        timeRestore,
        ...restOfDashboardChanges
      } = dashboardChanges;
      const dashboardBackupState = {
        // always back up view mode. This allows us to know which Dashboards were last changed while in edit mode.
        viewMode,
        ...restOfDashboardChanges
      };

      // Backup latest state from children that have unsaved changes
      if (hasChildrenUnsavedChanges || hasControlGroupChanges || hasLayoutChanges) {
        const {
          panels,
          references
        } = layoutManager.internalApi.serializeLayout();
        const {
          controlGroupInput,
          controlGroupReferences
        } = controlGroupManager.internalApi.serializeControlGroup();
        // dashboardStateToBackup.references will be used instead of savedObjectResult.references
        // To avoid missing references, make sure references contains all references
        // even if panels or control group does not have unsaved changes
        dashboardBackupState.references = [...(references !== null && references !== void 0 ? references : []), ...controlGroupReferences];
        if (hasChildrenUnsavedChanges) dashboardBackupState.panels = panels;
        if (hasControlGroupChanges) dashboardBackupState.controlGroupInput = controlGroupInput;
      }
      (0, _dashboard_backup_service.getDashboardBackupService)().setState(savedObjectId$.value, dashboardBackupState);
    }
  });
  const getLastSavedStateForChild = childId => {
    const lastSavedDashboardState = lastSavedState$.value;
    if (childId === _control_group_manager.CONTROL_GROUP_EMBEDDABLE_ID) {
      return lastSavedDashboardState.controlGroupInput ? {
        rawState: lastSavedDashboardState.controlGroupInput,
        references: getReferences(_control_group_manager.CONTROL_GROUP_EMBEDDABLE_ID)
      } : undefined;
    }
    return layoutManager.internalApi.getLastSavedStateForPanel(childId);
  };
  return {
    api: {
      asyncResetToLastSavedState: async () => {
        var _controlGroupManager$;
        const savedState = lastSavedState$.value;
        layoutManager.internalApi.reset();
        unifiedSearchManager.internalApi.reset(savedState);
        settingsManager.internalApi.reset(savedState);
        await ((_controlGroupManager$ = controlGroupManager.api.controlGroupApi$.value) === null || _controlGroupManager$ === void 0 ? void 0 : _controlGroupManager$.resetUnsavedChanges());
      },
      hasUnsavedChanges$,
      lastSavedStateForChild$: panelId => lastSavedState$.pipe((0, _rxjs.map)(() => getLastSavedStateForChild(panelId))),
      getLastSavedStateForChild
    },
    cleanup: () => {
      unsavedChangesSubscription.unsubscribe();
    },
    internalApi: {
      getLastSavedState: () => lastSavedState$.value,
      onSave: savedState => {
        lastSavedState$.next(savedState);
      }
    }
  };
}