"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scrollToTop = exports.scrollToBottom = exports.scrollTo = exports.scrollBy = exports.isAtBottomOfPage = exports.getViewportHeight = exports.getViewportBoundaries = exports.getScrollPosition = exports.getScrollDimensions = exports.getScrollContainer = void 0;
var _coreChromeLayoutConstants = require("@kbn/core-chrome-layout-constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Gets the main scroll container element for the application.
 * @returns The scroll container element (either the app scroll container or document.documentElement for window scroll)
 */
const getScrollContainer = () => {
  const appScroll = document.getElementById(_coreChromeLayoutConstants.APP_MAIN_SCROLL_CONTAINER_ID);
  if (appScroll instanceof HTMLElement) {
    return appScroll;
  }
  return document.documentElement;
};

/**
 * Gets the visible height of a scroll container's viewport.
 * @param container - The container to measure. Defaults to the main application scroll container
 * @returns The viewport height in pixels
 */
exports.getScrollContainer = getScrollContainer;
const getViewportHeight = (container = getScrollContainer()) => {
  return container.clientHeight;
};

/**
 * Gets the vertical boundaries of a scroll container's viewport.
 * Useful for checking if elements are visible within the viewport.
 * @param container - The container to measure. Defaults to the main application scroll container
 * @returns An object with top and bottom pixel values relative to the document
 */
exports.getViewportHeight = getViewportHeight;
const getViewportBoundaries = (container = getScrollContainer()) => {
  const rect = container.getBoundingClientRect();
  return {
    top: rect.top,
    bottom: rect.top + container.clientHeight
  };
};

/**
 * Gets the current scroll position of a container.
 * @param container - The container to measure. Defaults to the main application scroll container
 * @returns The current vertical scroll position in pixels
 */
exports.getViewportBoundaries = getViewportBoundaries;
const getScrollPosition = (container = getScrollContainer()) => {
  return container.scrollTop;
};

/**
 * Scrolls a container to a specific vertical position.
 * @param opts - Scroll options
 * @param opts.top - The vertical position to scroll to in pixels
 * @param opts.behavior - The scroll behavior ('auto' or 'smooth'). Default is 'auto'
 * @param container - The container to scroll. Defaults to the main application scroll container
 */
exports.getScrollPosition = getScrollPosition;
const scrollTo = (opts, container = getScrollContainer()) => {
  container.scrollTo({
    top: opts.top,
    behavior: opts.behavior
  });
};

/**
 * Scrolls a container to the top.
 * @param opts - Scroll options
 * @param opts.behavior - The scroll behavior ('auto' or 'smooth'). Default is 'auto'
 * @param container - The container to scroll. Defaults to the main application scroll container
 */
exports.scrollTo = scrollTo;
const scrollToTop = (opts = {}, container = getScrollContainer()) => {
  scrollTo({
    top: 0,
    behavior: opts.behavior
  }, container);
};

/**
 * Scrolls a container to the bottom.
 * @param opts - Scroll options
 * @param opts.behavior - The scroll behavior ('auto' or 'smooth'). Default is 'auto'
 * @param container - The container to scroll. Defaults to the main application scroll container
 */
exports.scrollToTop = scrollToTop;
const scrollToBottom = (opts = {}, container = getScrollContainer()) => {
  scrollTo({
    top: container.scrollHeight,
    behavior: opts.behavior
  }, container);
};

/**
 * Gets all scroll dimensions of a container at once for efficiency.
 * @param container - The container to measure. Defaults to the main application scroll container
 * @returns An object with scrollTop, scrollHeight, and clientHeight
 */
exports.scrollToBottom = scrollToBottom;
const getScrollDimensions = (container = getScrollContainer()) => {
  return {
    scrollTop: container.scrollTop,
    scrollHeight: container.scrollHeight,
    clientHeight: container.clientHeight
  };
};

/**
 * Scrolls a container by a relative amount.
 * @param opts - Scroll options
 * @param opts.top - The number of pixels to scroll (positive = down, negative = up)
 * @param opts.behavior - The scroll behavior ('auto' or 'smooth'). Default is 'auto'
 * @param container - The container to scroll. Defaults to the main application scroll container
 */
exports.getScrollDimensions = getScrollDimensions;
const scrollBy = (opts, container = getScrollContainer()) => {
  container.scrollBy({
    top: opts.top,
    behavior: opts.behavior
  });
};

/**
 * Detects if a scroll container has reached the bottom of its scrollable area.
 * @param container - The container to check. Defaults to the main application scroll container
 * @returns true if the container is scrolled to the bottom, false otherwise
 */
exports.scrollBy = scrollBy;
const isAtBottomOfPage = (container = getScrollContainer()) => {
  const {
    scrollTop,
    scrollHeight,
    clientHeight
  } = getScrollDimensions(container);
  return scrollHeight - clientHeight - scrollTop <= 1; // Allow 1px tolerance
};
exports.isAtBottomOfPage = isAtBottomOfPage;