"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CspPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _pRetry = _interopRequireDefault(require("p-retry"));
var _common = require("@kbn/fleet-plugin/common");
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _semver = _interopRequireDefault(require("semver"));
var _helpers = require("../common/utils/helpers");
var _subscription = require("../common/utils/subscription");
var _setup_routes = require("./routes/setup_routes");
var _saved_objects = require("./saved_objects");
var _data_views = require("./saved_objects/data_views");
var _create_indices = require("./create_indices/create_indices");
var _create_transforms = require("./create_transforms/create_transforms");
var _fleet_integration = require("./fleet_integration/fleet_integration");
var _constants = require("../common/constants");
var _findings_stats_task = require("./tasks/findings_stats_task");
var _register = require("./lib/telemetry/collectors/register");
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldSet(s, a, r) { return s.set(_assertClassBrand(s, a), r), r; }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
var _isInitialized = /*#__PURE__*/new WeakMap();
class CspPlugin {
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "isCloudEnabled", void 0);
    /**
     * CSP is initialized when the Fleet package is installed.
     * either directly after installation, or
     * when the plugin is started and a package is present.
     */
    _classPrivateFieldInitSpec(this, _isInitialized, false);
    // For integration versions earlier than 2.00, we will manually create an index alias for the deprecated latest index 'logs-cloud_security_posture.findings_latest-default'.
    // For integration versions 2.00 and above, the index alias will be automatically created or updated as part of the Transform setup.
    (0, _defineProperty2.default)(this, "initializeIndexAlias", async (esClient, logger) => {
      const isAliasExists = await esClient.indices.existsAlias({
        name: _cloudSecurityPostureCommon.CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_ALIAS
      });
      const isDeprecatedLatestIndexExists = await esClient.indices.exists({
        index: _cloudSecurityPostureCommon.DEPRECATED_CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_PATTERN
      });

      // This handles the following scenarios:
      // 1. A customer using an older integration version (pre-2.00) who has upgraded their Kibana stack.
      // 2. A customer with a new Kibana stack who installs an integration version earlier than 2.00 for the first time (e.g., in a serverless environment).
      if (isDeprecatedLatestIndexExists && !isAliasExists) {
        try {
          await esClient.indices.updateAliases({
            actions: [{
              add: {
                index: _cloudSecurityPostureCommon.DEPRECATED_CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_PATTERN,
                alias: _cloudSecurityPostureCommon.CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_ALIAS,
                is_write_index: true
              }
            }]
          });
          this.logger.info(`Index alias ${_cloudSecurityPostureCommon.CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_ALIAS} created successfully`);
        } catch (error) {
          this.logger.error(`Failed to create index alias ${_cloudSecurityPostureCommon.CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_ALIAS}`, error);
          throw error;
        }
      }
    });
    this.logger = initializerContext.logger.get();
    this.config = initializerContext.config.get();
  }
  setup(core, plugins) {
    core.savedObjects.registerType(_saved_objects.cspBenchmarkRule);
    core.savedObjects.registerType(_saved_objects.cspSettings);
    (0, _setup_routes.setupRoutes)({
      core,
      logger: this.logger,
      isPluginInitialized: () => _classPrivateFieldGet(_isInitialized, this)
    });
    const coreStartServices = core.getStartServices();
    this.setupCspTasks(plugins.taskManager, coreStartServices, this.logger);
    (0, _register.registerCspmUsageCollector)(this.logger, coreStartServices, plugins.usageCollection);
    this.isCloudEnabled = plugins.cloud.isCloudEnabled;
    return {};
  }
  start(core, plugins) {
    plugins.fleet.fleetSetupCompleted().then(async () => {
      const packageInfo = await (0, _pRetry.default)(() => plugins.fleet.packageService.asInternalUser.getInstallation(_constants.CLOUD_SECURITY_POSTURE_PACKAGE_NAME), getRetryOptions(this.logger, 'getInstallation'));

      // If package is installed we want to make sure all needed assets are installed
      if (packageInfo) {
        this.initialize(core, plugins.taskManager, packageInfo.install_version).catch(() => {});
      }
      plugins.fleet.registerExternalCallback('packagePolicyCreate', async packagePolicy => {
        var _packagePolicy$packag;
        const license = await plugins.licensing.getLicense();
        if ((0, _helpers.isCspPackage)((_packagePolicy$packag = packagePolicy.package) === null || _packagePolicy$packag === void 0 ? void 0 : _packagePolicy$packag.name)) {
          if (!(0, _subscription.isSubscriptionAllowed)(this.isCloudEnabled, license)) {
            throw new _common.FleetError('To use this feature you must upgrade your subscription or start a trial');
          }
          if (!isSingleEnabledInput(packagePolicy.inputs)) {
            throw new _common.FleetError('Only one enabled input is allowed per policy');
          }
        }
        return packagePolicy;
      });
      plugins.fleet.registerExternalCallback('packagePolicyCreate', async (packagePolicy, soClient) => {
        var _packagePolicy$packag2;
        if ((0, _helpers.isCspPackage)((_packagePolicy$packag2 = packagePolicy.package) === null || _packagePolicy$packag2 === void 0 ? void 0 : _packagePolicy$packag2.name)) {
          return (0, _helpers.cleanupCredentials)(packagePolicy);
        }
        return packagePolicy;
      });
      plugins.fleet.registerExternalCallback('packagePolicyUpdate', async (packagePolicy, soClient, esClient) => {
        var _packagePolicy$packag3;
        if ((0, _helpers.isCspPackage)((_packagePolicy$packag3 = packagePolicy.package) === null || _packagePolicy$packag3 === void 0 ? void 0 : _packagePolicy$packag3.name)) {
          const isIntegrationVersionIncludesTransformAsset = isTransformAssetIncluded(packagePolicy.package.version);
          await (0, _create_transforms.deletePreviousTransformsVersions)(esClient, isIntegrationVersionIncludesTransformAsset, this.logger);
          return (0, _helpers.cleanupCredentials)(packagePolicy);
        }
        return packagePolicy;
      });
      plugins.fleet.registerExternalCallback('packagePolicyPostCreate', async (packagePolicy, soClient) => {
        var _packagePolicy$packag4;
        if ((0, _helpers.isCspPackage)((_packagePolicy$packag4 = packagePolicy.package) === null || _packagePolicy$packag4 === void 0 ? void 0 : _packagePolicy$packag4.name)) {
          await this.initialize(core, plugins.taskManager, packagePolicy.package.version);
          return packagePolicy;
        }
        return packagePolicy;
      });
      plugins.fleet.registerExternalCallback('packagePolicyPostDelete', async deletedPackagePolicies => {
        for (const deletedPackagePolicy of deletedPackagePolicies) {
          var _deletedPackagePolicy;
          if ((0, _helpers.isCspPackage)((_deletedPackagePolicy = deletedPackagePolicy.package) === null || _deletedPackagePolicy === void 0 ? void 0 : _deletedPackagePolicy.name)) {
            const soClient = core.savedObjects.createInternalRepository();
            const packagePolicyService = plugins.fleet.packagePolicyService;
            const isPackageExists = await (0, _fleet_integration.isCspPackagePolicyInstalled)(packagePolicyService, soClient, this.logger);
            if (!isPackageExists) {
              await this.uninstallResources(plugins.taskManager, this.logger);
            }
          }
        }
      });
    }).catch(err => {
      this.logger.error('CSP plugin getInstallation operation failed after all retries', err);
    });
    return {};
  }
  stop() {}

  /**
   * Initialization is idempotent and required for (re)creating indices and transforms.
   */
  async initialize(core, taskManager, packagePolicyVersion) {
    this.logger.debug('initialize');
    const esClient = core.elasticsearch.client.asInternalUser;
    const soClient = core.savedObjects.createInternalRepository();
    const isIntegrationVersionIncludesTransformAsset = isTransformAssetIncluded(packagePolicyVersion);
    await (0, _create_indices.initializeCspIndices)(esClient, this.config, isIntegrationVersionIncludesTransformAsset, this.logger);
    await (0, _create_transforms.initializeCspTransforms)(esClient, isIntegrationVersionIncludesTransformAsset, this.logger);
    await (0, _findings_stats_task.scheduleFindingsStatsTask)(taskManager, this.logger);
    await this.initializeIndexAlias(esClient, this.logger);

    // Delete old and legacy CDR data views for all spaces
    await (0, _data_views.deleteOldAndLegacyCdrDataViewsForAllSpaces)(soClient, this.logger);
    _classPrivateFieldSet(_isInitialized, this, true);
  }
  async uninstallResources(taskManager, logger) {
    await (0, _findings_stats_task.removeFindingsStatsTask)(taskManager, logger);
  }
  setupCspTasks(taskManager, coreStartServices, logger) {
    (0, _findings_stats_task.setupFindingsStatsTask)(taskManager, coreStartServices, logger);
  }
}
exports.CspPlugin = CspPlugin;
const isSingleEnabledInput = inputs => inputs.filter(i => i.enabled).length === 1;
const isTransformAssetIncluded = integrationVersion => {
  const majorVersion = _semver.default.major(integrationVersion);
  return majorVersion >= 3;
};
const getRetryOptions = (logger, operation) => {
  return {
    retries: 3,
    onFailedAttempt: err => {
      const message = `CSP plugin ${operation} operation failed and will be retried: ${err.retriesLeft} more times; error: ${err.message}`;
      logger.warn(message);
    }
  };
};