"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCspmCloudConnectorUsageStats = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Checks if the cloud connector has valid credentials based on cloud provider
 */
const hasValidCredentials = (cloudProvider, vars) => {
  var _vars$role_arn, _vars$external_id, _vars$client_id, _vars$tenant_id, _vars$azure_cloud_con, _vars$credentials, _vars$credentials_fil;
  switch (cloudProvider) {
    case 'aws':
      return !!((_vars$role_arn = vars.role_arn) !== null && _vars$role_arn !== void 0 && _vars$role_arn.value && (_vars$external_id = vars.external_id) !== null && _vars$external_id !== void 0 && _vars$external_id.value);
    case 'azure':
      return !!((_vars$client_id = vars.client_id) !== null && _vars$client_id !== void 0 && _vars$client_id.value && (_vars$tenant_id = vars.tenant_id) !== null && _vars$tenant_id !== void 0 && _vars$tenant_id.value && (_vars$azure_cloud_con = vars.azure_cloud_connector_id) !== null && _vars$azure_cloud_con !== void 0 && _vars$azure_cloud_con.value);
    case 'gcp':
      return !!((_vars$credentials = vars.credentials) !== null && _vars$credentials !== void 0 && _vars$credentials.value || (_vars$credentials_fil = vars.credentials_file) !== null && _vars$credentials_fil !== void 0 && _vars$credentials_fil.value);
    default:
      return false;
  }
};
const getCspmCloudConnectorUsageStats = async (soClient, coreServices, logger) => {
  try {
    const [, cspServerPluginStartDeps] = await coreServices;

    // Get cloud connector service
    const cloudConnectorService = cspServerPluginStartDeps.fleet.cloudConnectorService;
    if (!cloudConnectorService) {
      logger.debug('Cloud connector service not available');
      return [];
    }

    // Get all cloud connectors
    const cloudConnectors = await cloudConnectorService.getList(soClient);
    if (!cloudConnectors.length) {
      logger.debug('No cloud connectors found');
      return [];
    }

    // Get package policies that use cloud connectors
    const packagePolicyService = cspServerPluginStartDeps.fleet.packagePolicyService;
    const packagePolicies = await packagePolicyService.list(soClient, {
      perPage: _common.SO_SEARCH_LIMIT,
      kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:"${_constants.CLOUD_SECURITY_POSTURE_PACKAGE_NAME}" AND ${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.cloud_connector_id:*`
    });
    const stats = cloudConnectors.map(connector => {
      // Filter package policies for this cloud connector
      const connectorPackagePolicies = packagePolicies.items.filter(policy => policy.cloud_connector_id === connector.id);

      // Extract integration types and packages
      const packagePolicyIds = [];
      connectorPackagePolicies.forEach(policy => {
        packagePolicyIds.push(policy.id);
      });
      return {
        id: connector.id,
        created_at: connector.created_at,
        updated_at: connector.updated_at,
        hasCredentials: hasValidCredentials(connector.cloudProvider, connector.vars || {}),
        cloud_provider: connector.cloudProvider,
        packagePolicyIds,
        packagePolicyCount: connectorPackagePolicies.length
      };
    });
    logger.info(`Collected CSPM cloud connector usage stats for ${stats.length} connectors`);
    return stats;
  } catch (error) {
    logger.error(`Failed to collect CSPM cloud connector usage stats: ${error.message}`);
    return [];
  }
};
exports.getCspmCloudConnectorUsageStats = getCspmCloudConnectorUsageStats;