"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.applyFilterlist = applyFilterlist;
exports.emptyStat = emptyStat;
exports.fieldNames = fieldNames;
exports.parseDiagnosticQueries = parseDiagnosticQueries;
exports.shouldExecute = shouldExecute;
var _crypto = require("crypto");
var YAML = _interopRequireWildcard(require("yaml"));
var _intervals = require("@kbn/task-manager-plugin/server/lib/intervals");
var _health_diagnostic_service = require("./health_diagnostic_service.types");
var _helpers = require("../helpers");
var _encryption = require("./encryption");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function shouldExecute(startDate, endDate, interval) {
  const nextDate = (0, _intervals.intervalFromDate)(startDate, interval);
  return nextDate !== undefined && nextDate < endDate;
}
function parseDiagnosticQueries(input) {
  return YAML.parseAllDocuments(input).map(doc => {
    return doc.toJSON();
  });
}
function fieldNames(documents) {
  const result = new Set();
  const traverse = (obj, path) => {
    if (Array.isArray(obj)) {
      if (obj.length > 0) {
        traverse(obj[0], `${path}[]`);
      } else {
        result.add(`${path}[]`);
      }
    } else if (obj && typeof obj === 'object') {
      for (const [key, value] of Object.entries(obj)) {
        traverse(value, path ? `${path}.${key}` : key);
      }
    } else {
      result.add(path);
    }
  };
  traverse(documents, '');
  return Array.from(result);
}
function emptyStat(name, now) {
  return {
    name,
    started: now.toISOString(),
    traceId: (0, _crypto.randomUUID)(),
    finished: new Date().toISOString(),
    numDocs: 0,
    passed: false,
    fieldNames: []
  };
}
async function applyFilterlist(data, rules, salt, query, encryptionPublicKeys) {
  const filteredResult = [];
  const hasEncryptAction = Object.values(rules).some(action => action === _health_diagnostic_service.Action.ENCRYPT);
  let dek;
  let encryptedDEK;
  let keyId;
  if (hasEncryptAction) {
    if (!(query !== null && query !== void 0 && query.encryptionKeyId)) {
      throw new Error('encryptionKeyId is required when filterlist contains encrypt actions');
    }
    if (!encryptionPublicKeys || !encryptionPublicKeys[query.encryptionKeyId]) {
      throw new Error(`Public key not found for encryptionKeyId: ${query.encryptionKeyId}`);
    }

    // same DEK for all the data
    keyId = query.encryptionKeyId;
    const publicKey = encryptionPublicKeys[keyId];
    dek = (0, _encryption.generateDEK)();
    encryptedDEK = (0, _encryption.encryptDEKWithRSA)(dek, publicKey);
  }
  const applyFilterToDoc = async doc => {
    const filteredDoc = {};
    for (const path of Object.keys(rules)) {
      const keys = path.split('.');
      await processPath(doc, filteredDoc, keys, path, 0);
    }
    return filteredDoc;
  };
  const processFieldValue = async (value, action) => {
    if (action === _health_diagnostic_service.Action.MASK) {
      return maskValue(String(value), salt);
    } else if (action === _health_diagnostic_service.Action.ENCRYPT) {
      if (!dek || !encryptedDEK || !keyId) {
        throw new Error('Encryption configuration not initialized');
      }
      return (0, _encryption.encryptField)(String(value), dek, encryptedDEK, keyId);
    } else if (action === _health_diagnostic_service.Action.KEEP) {
      return value;
    } else {
      throw new Error(`Unknown action: ${action}`);
    }
  };
  const processArrayField = async (nextValue, dst, key, keys, fullPath, keyIndex) => {
    if (!dst[key]) {
      dst[key] = [];
    }
    const dstArray = dst[key];
    for (let i = 0; i < nextValue.length; i++) {
      const item = nextValue[i];
      if (item && typeof item === 'object') {
        if (!dstArray[i]) {
          dstArray[i] = {};
        }
        await processPath(item, dstArray[i], keys, fullPath, keyIndex + 1);
      }
    }
  };
  const processPath = async (src, dst, keys, fullPath, keyIndex) => {
    if (keyIndex >= keys.length || !src || typeof src !== 'object') return;
    const key = keys[keyIndex];
    const srcObj = src;
    if (!Object.hasOwn(srcObj, key)) return;
    if (keyIndex === keys.length - 1) {
      const value = srcObj[key];
      const action = rules[fullPath];
      dst[key] = await processFieldValue(value, action);
    } else {
      const nextValue = srcObj[key];
      if (Array.isArray(nextValue)) {
        await processArrayField(nextValue, dst, key, keys, fullPath, keyIndex);
      } else if (nextValue && typeof nextValue === 'object') {
        dst[key] ??= {};
        await processPath(nextValue, dst[key], keys, fullPath, keyIndex + 1);
      }
    }
  };
  for (const rawDoc of data) {
    const doc = (0, _helpers.unflatten)(rawDoc);
    if (Array.isArray(doc)) {
      const docs = doc;
      const result = await Promise.all(docs.map(d => {
        return applyFilterToDoc(d);
      }));
      filteredResult.push(result);
    } else {
      filteredResult.push(await applyFilterToDoc(doc));
    }
  }
  return filteredResult;
}
async function maskValue(value, salt) {
  const encoder = new TextEncoder();
  const data = encoder.encode(salt + value);
  const hashBuffer = await crypto.subtle.digest('SHA-256', data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
}