"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QRadarProcessor = void 0;
var _transforms = require("./transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Processor for QRadar MITRE mappings enhancement
 */
class QRadarProcessor {
  constructor(context) {
    this.context = context;
  }
  getProcessor(type) {
    switch (type) {
      case 'mitre':
        return this.processMitreMappings.bind(this);
      case 'rules':
        return this.processRules.bind(this);
      default:
        throw new Error(`Unsupported QRadar processor type: ${type}`);
    }
  }
  processRules(rules) {
    return rules.map(rule => ({
      original_rule: (0, _transforms.transformQRadarRuleToOriginalRule)(rule),
      migration_id: this.context.migrationId
    }));
  }

  /**
   * Processes QRadar MITRE mappings and updates corresponding rules
   */
  async processMitreMappings(data) {
    const {
      migrationId,
      dataClient,
      logger
    } = this.context;
    try {
      // Extract rule titles from the mappings data
      const ruleTitles = Object.keys(data);
      if (ruleTitles.length === 0) {
        return [];
      }
      logger.debug(`Processing MITRE mappings for ${ruleTitles.length} QRadar rules in migration ${migrationId}`);

      // Fetch rules by titles
      const {
        data: rules
      } = await dataClient.get(migrationId, {
        filters: {
          titles: ruleTitles
        },
        from: 0,
        size: ruleTitles.length
      });
      if (rules.length === 0) {
        logger.warn(`No rules found matching provided titles for migration ${migrationId}`);
        return [];
      }
      const rulesToUpdate = [];
      for (const rule of rules) {
        const ruleTitle = rule.original_rule.title;
        const mitreMapping = data[ruleTitle];
        if (mitreMapping) {
          const threat = (0, _transforms.transformMitreMapping)(mitreMapping);
          rulesToUpdate.push({
            ...rule,
            original_rule: {
              ...rule.original_rule,
              threat
            }
          });
        }
      }
      logger.info(`Prepared Rules : ${JSON.stringify(rulesToUpdate, null, 2)}`);
      return rulesToUpdate;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      logger.error(`Error processing QRadar MITRE mappings: ${errorMessage}`);
      throw error;
    }
  }
}
exports.QRadarProcessor = QRadarProcessor;