"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mapSplunkSeverityToElasticSeverity = exports.mapRangedSeverityToElasticSeverity = exports.mapQradarSeverityToElasticSeverity = exports.getElasticSeverityFromOriginalRule = exports.getElasticRiskScoreFromOriginalRule = exports.getElasticRiskScoreFromElasticSeverity = void 0;
var _rules_xml = require("../../../../../../../../../../common/siem_migrations/parsers/qradar/rules_xml");
var _constants = require("../../../../../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const mapSplunkSeverityToElasticSeverity = splunkSeverity => {
  if (!splunkSeverity) {
    return _constants.DEFAULT_TRANSLATION_SEVERITY;
  }
  return _constants.SPLUNK_ELASTIC_ALERT_SEVERITY_MAP[splunkSeverity] || _constants.DEFAULT_TRANSLATION_SEVERITY;
};
exports.mapSplunkSeverityToElasticSeverity = mapSplunkSeverityToElasticSeverity;
const mapRangedSeverityToElasticSeverity = (severityValue, {
  min,
  max
}) => {
  // we normal min max to 1-10 scale
  const normalizedSeverity = Math.round((severityValue - min) / (max - min) * 9 + 1);
  if (normalizedSeverity <= 3) {
    return 'low';
  } else if (normalizedSeverity <= 6) {
    return 'medium';
  } else if (normalizedSeverity <= 8) {
    return 'high';
  } else {
    return 'critical';
  }
};
exports.mapRangedSeverityToElasticSeverity = mapRangedSeverityToElasticSeverity;
const mapQradarSeverityToElasticSeverity = async query => {
  const qradarXmlParser = new _rules_xml.QradarRulesXmlParser(query);
  try {
    const qradarSeverity = await qradarXmlParser.parseSeverityFromRuleData(query);
    if (qradarSeverity) {
      return mapRangedSeverityToElasticSeverity(parseInt(qradarSeverity, 10), {
        min: 1,
        max: 10
      });
    } else {
      return _constants.DEFAULT_TRANSLATION_SEVERITY;
    }
  } catch (e) {
    return _constants.DEFAULT_TRANSLATION_SEVERITY;
  }
};
exports.mapQradarSeverityToElasticSeverity = mapQradarSeverityToElasticSeverity;
const getElasticSeverityFromOriginalRule = async originalRule => {
  if (originalRule.query_language === 'spl' || originalRule.vendor === 'splunk') {
    return mapSplunkSeverityToElasticSeverity(originalRule.severity);
  } else if (originalRule.vendor === 'qradar') {
    const qradarXmlParser = new _rules_xml.QradarRulesXmlParser(originalRule.query);
    try {
      const qradarSeverity = await qradarXmlParser.parseSeverityFromRuleData(originalRule.query);
      if (qradarSeverity) {
        return mapRangedSeverityToElasticSeverity(parseInt(qradarSeverity, 10), {
          min: 1,
          max: 10
        });
      } else {
        return _constants.DEFAULT_TRANSLATION_SEVERITY;
      }
    } catch (e) {
      return _constants.DEFAULT_TRANSLATION_SEVERITY;
    }
  } else {
    return _constants.DEFAULT_TRANSLATION_SEVERITY;
  }
};
exports.getElasticSeverityFromOriginalRule = getElasticSeverityFromOriginalRule;
const getElasticRiskScoreFromElasticSeverity = elasticSeverity => {
  return _constants.ELASTIC_SEVERITY_TO_RISK_SCORE_MAP[elasticSeverity];
};
exports.getElasticRiskScoreFromElasticSeverity = getElasticRiskScoreFromElasticSeverity;
const getElasticRiskScoreFromOriginalRule = async originalRule => {
  const elasticSeverity = await getElasticSeverityFromOriginalRule(originalRule);
  return getElasticRiskScoreFromElasticSeverity(elasticSeverity);
};
exports.getElasticRiskScoreFromOriginalRule = getElasticRiskScoreFromOriginalRule;