"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemRuleMigrationsCreateQRadarRulesRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../../../../common/siem_migrations/constants");
var _rule_migration = require("../../../../../../../common/siem_migrations/model/api/rules/rule_migration.gen");
var _rules_xml = require("../../../../../../../common/siem_migrations/parsers/qradar/rules_xml");
var _audit = require("../../../../common/api/util/audit");
var _authz = require("../../../../common/api/util/authz");
var _with_existing_migration_id = require("../../../../common/api/util/with_existing_migration_id");
var _with_license = require("../../../../common/api/util/with_license");
var _get_vendor_processor = require("../../../vendors/get_vendor_processor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemRuleMigrationsCreateQRadarRulesRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.SIEM_RULE_MIGRATION_QRADAR_RULES_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    },
    options: {
      body: {
        maxBytes: 25 * 1024 * 1024
      }
    } // rise payload limit to 25MB
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.CreateQRadarRuleMigrationRulesRequestBody),
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.CreateQRadarRuleMigrationRulesRequestParams)
      }
    }
  }, (0, _with_license.withLicense)((0, _with_existing_migration_id.withExistingMigration)(async (context, req, res) => {
    const {
      migration_id: migrationId
    } = req.params;
    const {
      xml
    } = req.body;
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution, 'rules');
    try {
      var _Object$keys;
      const ctx = await context.resolve(['securitySolution']);
      const ruleMigrationsClient = ctx.securitySolution.siemMigrations.getRulesClient();

      // Parse QRadar XML
      const parser = new _rules_xml.QradarRulesXmlParser(xml);
      const qradarRules = await parser.getRules();
      const qRadarResources = await parser.getResources();
      if (!qradarRules || qradarRules.length === 0) {
        return res.badRequest({
          body: {
            message: 'No rules found in the provided XML'
          }
        });
      }
      const rulesCount = qradarRules.length;
      const VendorProcessor = (0, _get_vendor_processor.getVendorProcessor)('qradar');
      const rulesProcessor = new VendorProcessor({
        migrationId,
        dataClient: ruleMigrationsClient.data.items,
        logger
      }).getProcessor('rules');
      const rulesToBeCreated = rulesProcessor(qradarRules);
      if (rulesCount === 0) {
        return res.badRequest({
          body: {
            message: 'No valid rules could be extracted from the XML'
          }
        });
      }
      const resourcesToBeCreated = [];
      if (((_Object$keys = Object.keys(qRadarResources)) === null || _Object$keys === void 0 ? void 0 : _Object$keys.length) > 0) {
        for (const resourceType of Object.keys(qRadarResources)) {
          const resourceTypeList = qRadarResources[resourceType];
          if (resourceTypeList && (resourceTypeList === null || resourceTypeList === void 0 ? void 0 : resourceTypeList.length) > 0) {
            const qRadarResourcesWithMigrationId = resourceTypeList.map(resource => ({
              ...resource,
              type: resourceType,
              migration_id: migrationId
            }));
            resourcesToBeCreated.push(...qRadarResourcesWithMigrationId);
          }
        }
        await ruleMigrationsClient.data.resources.create(resourcesToBeCreated);
      }
      await siemMigrationAuditLogger.logAddRules({
        migrationId,
        count: rulesCount
      });
      await ruleMigrationsClient.data.items.create(rulesToBeCreated);

      // TODO: Handle and report success and failures
      // For example,  {success: number, failed: number, errors: Array<{ruleId: string, error: string}>}
      return res.ok({
        body: {
          message: `Successfully imported ${rulesCount} QRadar rule${rulesCount !== 1 ? 's' : ''}`
        }
      });
    } catch (error) {
      logger.error(`Error processing QRadar XML: ${error.message}`);
      await siemMigrationAuditLogger.logAddRules({
        migrationId,
        count: 0,
        error
      });
      return res.badRequest({
        body: {
          message: `Failed to parse QRadar XML: ${error.message}`
        }
      });
    }
  })));
};
exports.registerSiemRuleMigrationsCreateQRadarRulesRoute = registerSiemRuleMigrationsCreateQRadarRulesRoute;