"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemRuleMigrationsGetRulesRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../../../common/siem_migrations/constants");
var _rule_migration = require("../../../../../../common/siem_migrations/model/api/rules/rule_migration.gen");
var _audit = require("../../../common/api/util/audit");
var _authz = require("../../../common/api/util/authz");
var _with_license = require("../../../common/api/util/with_license");
var _with_existing_migration_id = require("../../../common/api/util/with_existing_migration_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemRuleMigrationsGetRulesRoute = (router, logger) => {
  router.versioned.get({
    path: _constants.SIEM_RULE_MIGRATION_RULES_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.GetRuleMigrationRulesRequestParams),
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.GetRuleMigrationRulesRequestQuery)
      }
    }
  }, (0, _with_license.withLicense)((0, _with_existing_migration_id.withExistingMigration)(async (context, req, res) => {
    const {
      migration_id: migrationId
    } = req.params;
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution, 'rules');
    try {
      const ctx = await context.resolve(['securitySolution']);
      const ruleMigrationsClient = ctx.securitySolution.siemMigrations.getRulesClient();
      const {
        page,
        per_page: size
      } = req.query;
      const options = {
        filters: {
          searchTerm: req.query.search_term,
          ids: req.query.ids,
          prebuilt: req.query.is_prebuilt,
          installed: req.query.is_installed,
          fullyTranslated: req.query.is_fully_translated,
          partiallyTranslated: req.query.is_partially_translated,
          untranslatable: req.query.is_untranslatable,
          failed: req.query.is_failed,
          missingIndex: req.query.is_missing_index
        },
        sort: {
          sortField: req.query.sort_field,
          sortDirection: req.query.sort_direction
        },
        size,
        from: page && size ? page * size : 0
      };
      const result = await ruleMigrationsClient.data.items.get(migrationId, options);
      await siemMigrationAuditLogger.logGetMigrationItems({
        migrationId
      });
      return res.ok({
        body: result
      });
    } catch (error) {
      logger.error(error);
      await siemMigrationAuditLogger.logGetMigrationItems({
        migrationId,
        error
      });
      return res.badRequest({
        body: error.message
      });
    }
  })));
};
exports.registerSiemRuleMigrationsGetRulesRoute = registerSiemRuleMigrationsGetRulesRoute;