"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemRuleMigrationsEnhanceRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../../../common/siem_migrations/constants");
var _rule_migration = require("../../../../../../common/siem_migrations/model/api/rules/rule_migration.gen");
var _authz = require("../../../common/api/util/authz");
var _audit = require("../../../common/api/util/audit");
var _with_license = require("../../../common/api/util/with_license");
var _with_existing_migration_id = require("../../../common/api/util/with_existing_migration_id");
var _get_vendor_processor = require("../../vendors/get_vendor_processor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemRuleMigrationsEnhanceRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.SIEM_RULE_MIGRATION_RULES_ENHANCE_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.RuleMigrationEnhanceRuleRequestParams),
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.RuleMigrationEnhanceRuleRequestBody)
      }
    }
  }, (0, _with_license.withLicense)((0, _with_existing_migration_id.withExistingMigration)(async (context, req, res) => {
    const {
      migration_id: migrationId
    } = req.params;
    const {
      vendor,
      type: enhancementType,
      data
    } = req.body;
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution, 'rules');
    try {
      const ctx = await context.resolve(['securitySolution']);
      const ruleMigrationsClient = ctx.securitySolution.siemMigrations.getRulesClient();
      const dataClient = ruleMigrationsClient.data.items;
      const VendorProcessor = (0, _get_vendor_processor.getVendorProcessor)(vendor);
      const processor = new VendorProcessor({
        migrationId,
        dataClient,
        logger
      }).getProcessor(enhancementType);
      await siemMigrationAuditLogger.logEnhanceRules({
        migrationId,
        vendor,
        enhancementType
      });

      // Process the enhancement data
      const rulesToBeUpdate = await processor(data);
      if (rulesToBeUpdate.length === 0) {
        return res.badRequest({
          body: {
            message: 'No rules to enhance'
          }
        });
      }
      await dataClient.update(rulesToBeUpdate);
      return res.ok({
        body: {
          updated: true
        }
      });
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      logger.error(`Error enhancing rules for migration ${migrationId}: ${errorMessage}`);
      await siemMigrationAuditLogger.logEnhanceRules({
        migrationId,
        vendor,
        enhancementType,
        error
      });
      return res.customError({
        statusCode: 500,
        body: {
          message: `Failed to enhance rules: ${errorMessage}`
        }
      });
    }
  })));
};
exports.registerSiemRuleMigrationsEnhanceRoute = registerSiemRuleMigrationsEnhanceRoute;