"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemRuleMigrationsResourceUpsertRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _lodash = require("lodash");
var _types = require("../../../../../../common/siem_migrations/rules/resources/types");
var _constants = require("../../../../../../common/siem_migrations/constants");
var _rule_migration = require("../../../../../../common/siem_migrations/model/api/rules/rule_migration.gen");
var _resources = require("../../../../../../common/siem_migrations/rules/resources");
var _audit = require("../../../common/api/util/audit");
var _authz = require("../../../common/api/util/authz");
var _lookups = require("../util/lookups");
var _with_license = require("../../../common/api/util/with_license");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemRuleMigrationsResourceUpsertRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.SIEM_RULE_MIGRATION_RESOURCES_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    },
    options: {
      body: {
        maxBytes: 26214400
      }
    } // rise payload limit to 25MB
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.UpsertRuleMigrationResourcesRequestParams),
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.UpsertRuleMigrationResourcesRequestBody)
      }
    }
  }, (0, _with_license.withLicense)(async (context, req, res) => {
    const resources = req.body;
    const migrationId = req.params.migration_id;
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution, 'rules');
    try {
      const ctx = await context.resolve(['securitySolution']);
      const ruleMigrationsClient = ctx.securitySolution.siemMigrations.getRulesClient();
      await siemMigrationAuditLogger.logUploadResources({
        migrationId
      });

      // Check if the migration exists
      const {
        data
      } = await ruleMigrationsClient.data.items.get(migrationId, {
        size: 1
      });
      const [rule] = data;
      if (!rule) {
        return res.notFound({
          body: {
            message: 'Migration not found'
          }
        });
      }
      const [lookups, macros] = (0, _lodash.partition)(resources, {
        type: 'lookup'
      });
      const processedLookups = await (0, _lookups.processLookups)(lookups, ruleMigrationsClient.data.lookups);
      const resourcesUpsert = [...macros, ...processedLookups].map(resource => ({
        ...resource,
        migration_id: migrationId
      }));

      // Upsert the resources
      await ruleMigrationsClient.data.resources.upsert(resourcesUpsert);
      if (!(0, _types.isResourceSupportedVendor)(rule.original_rule.vendor)) {
        logger.debug(`Identifying resources for rule migration [id=${migrationId}] and vendor [${rule.original_rule.vendor}]  is not supported. Skipping resource identification.`);
        return res.ok({
          body: {
            acknowledged: true
          }
        });
      }
      const resourceIdentifier = new _resources.RuleResourceIdentifier(rule.original_rule.vendor);
      const resourcesToCreate = resourceIdentifier.fromResources(resources).map(resource => ({
        ...resource,
        migration_id: migrationId
      }));
      await ruleMigrationsClient.data.resources.create(resourcesToCreate);
      return res.ok({
        body: {
          acknowledged: true
        }
      });
      // Create identified resource documents to keep track of them (without content)
    } catch (error) {
      logger.error(error);
      await siemMigrationAuditLogger.logUploadResources({
        migrationId,
        error
      });
      return res.customError({
        statusCode: 500,
        body: error.message
      });
    }
  }));
};
exports.registerSiemRuleMigrationsResourceUpsertRoute = registerSiemRuleMigrationsResourceUpsertRoute;