"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.applyCriticalityModifier = void 0;
var _helpers = require("../helpers");
var _helpers2 = require("../../asset_criticality/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const applyCriticalityModifier = async ({
  page,
  deps,
  globalWeight
}) => {
  if (page.buckets.length === 0) {
    return [];
  }
  const identifiers = page.buckets.map(bucket => ({
    id_field: page.identifierField,
    id_value: bucket.key[page.identifierField]
  }));
  const criticalities = await deps.assetCriticalityService.getCriticalitiesByIdentifiers(identifiers).catch(error => {
    deps.logger.warn(`Error retrieving criticality: ${error}. Scoring will proceed without criticality information.`);
    return [];
  });
  return page.buckets.map(bucket => {
    const criticality = criticalities.find(c => c.id_field === page.identifierField && c.id_value === bucket.key[page.identifierField]);
    return calculateScoreAndContributions(bucket.top_inputs.risk_details.value.normalized_score, criticality, globalWeight);
  });
};
exports.applyCriticalityModifier = applyCriticalityModifier;
const calculateScoreAndContributions = (normalizedBaseScore, criticality, globalWeight) => {
  const criticalityModifier = (0, _helpers2.getCriticalityModifier)(criticality === null || criticality === void 0 ? void 0 : criticality.criticality_level);
  if (!criticalityModifier) {
    return {
      category_2_score: 0,
      category_2_count: 0
    };
  }
  const weightedNormalizedScore = globalWeight !== undefined ? normalizedBaseScore * globalWeight : normalizedBaseScore;
  const updatedNormalizedScore = (0, _helpers2.bayesianUpdate)({
    modifier: criticalityModifier,
    score: weightedNormalizedScore
  });
  const contribution = updatedNormalizedScore - weightedNormalizedScore;
  return {
    category_2_score: (0, _helpers.max10DecimalPlaces)(contribution),
    category_2_count: 1,
    // modifier exists, so count as 1
    criticality_level: criticality === null || criticality === void 0 ? void 0 : criticality.criticality_level,
    criticality_modifier: criticalityModifier
  };
};