"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processScores = exports.getGlobalWeightForIdentifierType = exports.filterFromRange = exports.calculateRiskScores = exports.buildFiltersForEntityType = void 0;
var _lodash = require("lodash");
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _esQuery = require("@kbn/es-query");
var _utils = require("../../../../common/entity_analytics/utils");
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _with_security_span = require("../../../utils/with_security_span");
var _helpers = require("../asset_criticality/helpers");
var _helpers2 = require("./helpers");
var _constants = require("./constants");
var _painless = require("./painless");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatForResponse = ({
  bucket,
  criticality,
  now,
  identifierField,
  includeNewFields,
  globalWeight
}) => {
  const riskDetails = bucket.top_inputs.risk_details;

  // Apply global weight to the score if provided
  const weightedScore = globalWeight !== undefined ? riskDetails.value.score * globalWeight : riskDetails.value.score;
  const weightedNormalizedScore = globalWeight !== undefined ? riskDetails.value.normalized_score * globalWeight : riskDetails.value.normalized_score;
  const criticalityModifier = (0, _helpers.getCriticalityModifier)(criticality === null || criticality === void 0 ? void 0 : criticality.criticality_level);
  const normalizedScoreWithCriticality = (0, _helpers.applyCriticalityToScore)({
    score: weightedNormalizedScore,
    modifier: criticalityModifier
  });
  const calculatedLevel = (0, _risk_engine.getRiskLevel)(normalizedScoreWithCriticality);
  const categoryTwoScore = normalizedScoreWithCriticality - weightedNormalizedScore;
  const categoryTwoCount = criticalityModifier ? 1 : 0;
  const newFields = {
    category_2_score: (0, _helpers2.max10DecimalPlaces)(categoryTwoScore),
    category_2_count: categoryTwoCount,
    criticality_level: criticality === null || criticality === void 0 ? void 0 : criticality.criticality_level,
    criticality_modifier: criticalityModifier
  };
  return {
    '@timestamp': now,
    id_field: identifierField,
    id_value: bucket.key[identifierField],
    calculated_level: calculatedLevel,
    calculated_score: (0, _helpers2.max10DecimalPlaces)(weightedScore),
    calculated_score_norm: (0, _helpers2.max10DecimalPlaces)(normalizedScoreWithCriticality),
    category_1_score: (0, _helpers2.max10DecimalPlaces)(riskDetails.value.category_1_score / _constants.RIEMANN_ZETA_VALUE),
    // normalize value to be between 0-100
    category_1_count: riskDetails.value.category_1_count,
    notes: riskDetails.value.notes,
    inputs: riskDetails.value.risk_inputs.map(riskInput => {
      var _riskInput$rule_name;
      return {
        id: riskInput.id,
        index: riskInput.index,
        description: `Alert from Rule: ${(_riskInput$rule_name = riskInput.rule_name) !== null && _riskInput$rule_name !== void 0 ? _riskInput$rule_name : 'RULE_NOT_FOUND'}`,
        category: _risk_engine.RiskCategories.category_1,
        risk_score: riskInput.score,
        timestamp: riskInput.time,
        contribution_score: riskInput.contribution
      };
    }),
    ...(includeNewFields ? newFields : {})
  };
};
const filterFromRange = range => ({
  range: {
    '@timestamp': {
      lt: range.end,
      gte: range.start
    }
  }
});

/**
 * Builds filters for a specific entity type, including entity-specific custom filters
 */
exports.filterFromRange = filterFromRange;
const buildFiltersForEntityType = (entityType, userFilter, customFilters = [], excludeAlertStatuses = [], excludeAlertTags = []) => {
  const filters = [{
    exists: {
      field: _technical_rule_data_field_names.ALERT_RISK_SCORE
    }
  }];

  // Add existing user filter (backward compatibility)
  if (!(0, _lodash.isEmpty)(userFilter)) {
    filters.push(userFilter);
  }

  // Add alert status exclusions
  if (excludeAlertStatuses.length > 0) {
    filters.push({
      bool: {
        must_not: {
          terms: {
            [_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS]: excludeAlertStatuses
          }
        }
      }
    });
  }

  // Add alert tag exclusions
  if (excludeAlertTags.length > 0) {
    filters.push({
      bool: {
        must_not: {
          terms: {
            [_technical_rule_data_field_names.ALERT_WORKFLOW_TAGS]: excludeAlertTags
          }
        }
      }
    });
  }

  // Add entity-specific custom filters (EXCLUSIVE - exclude matching alerts)
  customFilters.filter(f => f.entity_types.includes(entityType)).forEach(f => {
    try {
      const esQuery = (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(f.filter));
      filters.push({
        bool: {
          must: esQuery
        }
      });
    } catch (error) {
      // Log warning but don't fail the entire query
      // Note: Invalid KQL filters are silently ignored to prevent query failures
    }
  });
  return filters;
};
exports.buildFiltersForEntityType = buildFiltersForEntityType;
const buildIdentifierTypeAggregation = ({
  afterKeys,
  identifierType,
  pageSize,
  weights,
  alertSampleSizePerShard,
  scriptedMetricPainless
}) => {
  const globalIdentifierTypeWeight = getGlobalWeightForIdentifierType(identifierType, weights);
  const identifierField = (0, _helpers2.getFieldForIdentifier)(identifierType);
  return {
    composite: {
      size: pageSize,
      sources: [{
        [identifierField]: {
          terms: {
            field: identifierField
          }
        }
      }],
      after: (0, _helpers2.getAfterKeyForIdentifierType)({
        identifierType,
        afterKeys
      })
    },
    aggs: {
      top_inputs: {
        sampler: {
          shard_size: alertSampleSizePerShard
        },
        aggs: {
          risk_details: {
            scripted_metric: {
              init_script: scriptedMetricPainless.init,
              map_script: scriptedMetricPainless.map,
              combine_script: scriptedMetricPainless.combine,
              params: {
                p: _constants.RIEMANN_ZETA_S_VALUE,
                risk_cap: _constants.RIEMANN_ZETA_VALUE,
                global_identifier_type_weight: globalIdentifierTypeWeight || 1
              },
              reduce_script: scriptedMetricPainless.reduce
            }
          }
        }
      }
    }
  };
};
const processScores = async ({
  assetCriticalityService,
  buckets,
  identifierField,
  logger,
  now,
  identifierType,
  weights
}) => {
  if (buckets.length === 0) {
    return [];
  }
  const identifiers = buckets.map(bucket => ({
    id_field: identifierField,
    id_value: bucket.key[identifierField]
  }));
  let criticalities = [];
  try {
    criticalities = await assetCriticalityService.getCriticalitiesByIdentifiers(identifiers);
  } catch (e) {
    logger.warn(`Error retrieving criticality: ${e}. Scoring will proceed without criticality information.`);
  }
  const globalWeight = identifierType ? getGlobalWeightForIdentifierType(identifierType, weights) : undefined;
  return buckets.map(bucket => {
    const criticality = criticalities.find(c => c.id_field === identifierField && c.id_value === bucket.key[identifierField]);
    return formatForResponse({
      bucket,
      criticality,
      identifierField,
      now,
      includeNewFields: true,
      globalWeight
    });
  });
};
exports.processScores = processScores;
const getGlobalWeightForIdentifierType = (identifierType, weights) => {
  var _weights$find;
  return weights === null || weights === void 0 ? void 0 : (_weights$find = weights.find(weight => weight.type === _risk_engine.RiskWeightTypes.global)) === null || _weights$find === void 0 ? void 0 : _weights$find[identifierType];
};
exports.getGlobalWeightForIdentifierType = getGlobalWeightForIdentifierType;
const calculateRiskScores = async ({
  afterKeys: userAfterKeys,
  assetCriticalityService,
  debug,
  esClient,
  filter: userFilter,
  identifierType,
  index,
  logger,
  pageSize,
  range,
  runtimeMappings,
  weights,
  alertSampleSizePerShard = 10_000,
  excludeAlertStatuses = [],
  experimentalFeatures,
  excludeAlertTags = [],
  filters: customFilters = []
}) => (0, _with_security_span.withSecuritySpan)('calculateRiskScores', async () => {
  var _combinedAggregations, _combinedAggregations2, _combinedAggregations3, _combinedAggregations4, _combinedAggregations5, _combinedAggregations6;
  const now = new Date().toISOString();
  const scriptedMetricPainless = await (0, _painless.getPainlessScripts)();
  const identifierTypes = identifierType ? [identifierType] : (0, _utils.getEntityAnalyticsEntityTypes)();

  // Build base filters that apply to all entity types
  const baseFilters = [filterFromRange(range), {
    exists: {
      field: _technical_rule_data_field_names.ALERT_RISK_SCORE
    }
  }];

  // Create separate queries for each entity type with entity-specific filters
  const entityQueries = identifierTypes.map(_identifierType => {
    // Build entity-specific filters
    const entityFilters = buildFiltersForEntityType(_identifierType, userFilter, customFilters, excludeAlertStatuses, excludeAlertTags);

    // Combine base filters with entity-specific filters
    const allFilters = [...baseFilters, ...entityFilters];
    return {
      entityType: _identifierType,
      request: {
        size: 0,
        _source: false,
        index,
        ignore_unavailable: true,
        runtime_mappings: runtimeMappings,
        query: {
          function_score: {
            query: {
              bool: {
                filter: allFilters,
                should: [{
                  match_all: {} // This forces ES to calculate score
                }]
              }
            },
            field_value_factor: {
              field: _technical_rule_data_field_names.ALERT_RISK_SCORE // sort by risk score
            }
          }
        },
        aggs: {
          [_identifierType]: buildIdentifierTypeAggregation({
            afterKeys: userAfterKeys,
            identifierType: _identifierType,
            pageSize,
            weights,
            alertSampleSizePerShard,
            scriptedMetricPainless
          })
        }
      }
    };
  });

  // Execute queries for each entity type
  const responses = await Promise.all(entityQueries.map(async ({
    entityType,
    request: entityRequest
  }) => {
    if (debug) {
      logger.info(`Executing Risk Score query for ${entityType}:\n${JSON.stringify(entityRequest)}`);
    }
    const response = await esClient.search(entityRequest);
    if (debug) {
      logger.info(`Received Risk Score response for ${entityType}:\n${JSON.stringify(response)}`);
    }
    return {
      entityType,
      response,
      request: entityRequest
    };
  }));

  // Combine results from all entity queries
  const combinedAggregations = {};
  const combinedAfterKeys = {};
  responses.forEach(({
    entityType,
    response
  }) => {
    if (response.aggregations && response.aggregations[entityType]) {
      var _entityType;
      combinedAggregations[entityType] = response.aggregations[entityType];
      combinedAfterKeys[entityType] = (_entityType = response.aggregations[entityType]) === null || _entityType === void 0 ? void 0 : _entityType.after_key;
    }
  });
  const userBuckets = (_combinedAggregations = (_combinedAggregations2 = combinedAggregations.user) === null || _combinedAggregations2 === void 0 ? void 0 : _combinedAggregations2.buckets) !== null && _combinedAggregations !== void 0 ? _combinedAggregations : [];
  const hostBuckets = (_combinedAggregations3 = (_combinedAggregations4 = combinedAggregations.host) === null || _combinedAggregations4 === void 0 ? void 0 : _combinedAggregations4.buckets) !== null && _combinedAggregations3 !== void 0 ? _combinedAggregations3 : [];
  const serviceBuckets = (_combinedAggregations5 = (_combinedAggregations6 = combinedAggregations.service) === null || _combinedAggregations6 === void 0 ? void 0 : _combinedAggregations6.buckets) !== null && _combinedAggregations5 !== void 0 ? _combinedAggregations5 : [];
  const afterKeys = {
    host: combinedAfterKeys.host,
    user: combinedAfterKeys.user,
    service: experimentalFeatures ? combinedAfterKeys.service : undefined
  };
  const hostScores = await processScores({
    assetCriticalityService,
    buckets: hostBuckets,
    identifierField: 'host.name',
    logger,
    now
  });
  const userScores = await processScores({
    assetCriticalityService,
    buckets: userBuckets,
    identifierField: 'user.name',
    logger,
    now
  });
  const serviceScores = await processScores({
    assetCriticalityService,
    buckets: serviceBuckets,
    identifierField: 'service.name',
    logger,
    now
  });
  return {
    ...(debug ? {
      debug: {
        request: JSON.stringify(responses.map(({
          entityType,
          request
        }) => ({
          entityType,
          request
        }))),
        response: JSON.stringify(responses.map(({
          entityType,
          response
        }) => ({
          entityType,
          response
        })))
      }
    } : {}),
    after_keys: afterKeys,
    scores: {
      host: hostScores,
      user: userScores,
      service: serviceScores
    }
  };
});
exports.calculateRiskScores = calculateRiskScores;