"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.riskScoreDocFactory = exports.applyScoreModifiers = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _constants = require("./constants");
var _calculate_risk_scores = require("./calculate_risk_scores");
var _asset_criticality = require("./modifiers/asset_criticality");
var _privileged_users = require("./modifiers/privileged_users");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const applyScoreModifiers = async ({
  now,
  identifierType,
  deps,
  weights,
  page,
  experimentalFeatures
}) => {
  const globalWeight = identifierType ? (0, _calculate_risk_scores.getGlobalWeightForIdentifierType)(identifierType, weights) : undefined;
  const modifierPromises = [(0, _asset_criticality.applyCriticalityModifier)({
    page,
    globalWeight,
    deps: _lodash.default.pick(deps, ['assetCriticalityService', 'logger'])
  }), (0, _privileged_users.applyPrivmonModifier)({
    page,
    globalWeight,
    experimentalFeatures,
    deps: _lodash.default.pick(deps, ['privmonUserCrudService', 'logger'])
  })];
  const [criticality, privmon] = await Promise.all(modifierPromises);
  return _lodash.default.zipWith(page.buckets, criticality, privmon, riskScoreDocFactory({
    now,
    identifierField: page.identifierField,
    globalWeight
  }));
};
exports.applyScoreModifiers = applyScoreModifiers;
const riskScoreDocFactory = ({
  now,
  identifierField,
  globalWeight = 1
}) => (bucket, criticalityFields, privmonFields) => {
  const risk = bucket.top_inputs.risk_details;
  const alertsRiskScoreFields = {
    category_1_score: (0, _helpers.max10DecimalPlaces)(risk.value.category_1_score / _constants.RIEMANN_ZETA_VALUE),
    // normalize value to be between 0-100
    category_1_count: risk.value.category_1_count
  };
  const totalScoreWithModifiers = risk.value.normalized_score * globalWeight + criticalityFields.category_2_score + privmonFields.category_3_score;
  const weightedScore = globalWeight !== undefined ? risk.value.score * globalWeight : risk.value.score;
  const finalRiskScoreFields = {
    calculated_level: (0, _risk_engine.getRiskLevel)(totalScoreWithModifiers),
    calculated_score: (0, _helpers.max10DecimalPlaces)(weightedScore),
    calculated_score_norm: (0, _helpers.max10DecimalPlaces)(totalScoreWithModifiers)
  };
  return {
    '@timestamp': now,
    id_field: identifierField,
    id_value: bucket.key[identifierField],
    ...finalRiskScoreFields,
    ...alertsRiskScoreFields,
    ...criticalityFields,
    ...privmonFields,
    notes: risk.value.notes,
    inputs: risk.value.risk_inputs.map(riskInput => {
      var _riskInput$rule_name;
      return {
        id: riskInput.id,
        index: riskInput.index,
        description: `Alert from Rule: ${(_riskInput$rule_name = riskInput.rule_name) !== null && _riskInput$rule_name !== void 0 ? _riskInput$rule_name : 'RULE_NOT_FOUND'}`,
        category: _risk_engine.RiskCategories.category_1,
        risk_score: riskInput.score,
        timestamp: riskInput.time,
        contribution_score: riskInput.contribution
      };
    })
  };
};
exports.riskScoreDocFactory = riskScoreDocFactory;