"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPrivilegedUsersCrudService = void 0;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Helper function to upsert a single API label into the merged labels array
const upsertApiLabel = (mergedLabels, apiLabel) => {
  // Skip labels with missing required fields
  if (!apiLabel.field || !apiLabel.value) {
    return;
  }
  const existingLabelIndex = mergedLabels.findIndex(label => label.field === apiLabel.field && label.source === 'api');
  if (existingLabelIndex >= 0) {
    // Update existing API label
    mergedLabels[existingLabelIndex] = {
      field: apiLabel.field,
      value: apiLabel.value,
      source: 'api'
    };
  } else {
    // Add new API label
    mergedLabels.push({
      field: apiLabel.field,
      value: apiLabel.value,
      source: 'api'
    });
  }
};

// Helper function to merge API labels with existing labels
const mergeApiLabels = (existingLabels, apiLabels) => {
  const mergedLabels = [...(existingLabels || [])];
  if (apiLabels) {
    for (const apiLabel of apiLabels) {
      upsertApiLabel(mergedLabels, apiLabel);
    }
  }
  return mergedLabels;
};
const createPrivilegedUsersCrudService = ({
  esClient,
  index,
  logger
}) => {
  const create = async (user, source, maxUsersAllowed) => {
    var _user$user, _user$entity_analytic2, _user$entity_analytic3;
    // This method handles two scenarios:
    // 1. If user exists: Update existing user with new labels and sources
    // 2. If user doesn't exist: Create a new user

    logger.info(`Maximum supported number of privileged users allowed: ${maxUsersAllowed}`);
    const timestamp = new Date().toISOString();
    // Check if user already exists by username
    const username = (_user$user = user.user) === null || _user$user === void 0 ? void 0 : _user$user.name;
    if (username) {
      const existingUserResponse = await esClient.search({
        index,
        query: {
          term: {
            'user.name': username
          }
        },
        size: 1
      });
      if (existingUserResponse.hits.hits.length > 0) {
        // Update existing user
        const existingUser = existingUserResponse.hits.hits[0];
        const existingUserId = existingUser._id;
        if (existingUserId) {
          var _existingUserDoc$labe, _user$entity_analytic, _existingUserDoc$enti, _existingUserDoc$even;
          const existingUserDoc = existingUser._source;
          const existingSources = (existingUserDoc === null || existingUserDoc === void 0 ? void 0 : (_existingUserDoc$labe = existingUserDoc.labels) === null || _existingUserDoc$labe === void 0 ? void 0 : _existingUserDoc$labe.sources) || [];
          const updatedSources = existingSources.includes(source) ? existingSources : [...existingSources, source];

          // Handle API labels if provided
          const apiLabels = (_user$entity_analytic = user.entity_analytics_monitoring) === null || _user$entity_analytic === void 0 ? void 0 : _user$entity_analytic.labels;
          const existingLabels = existingUserDoc === null || existingUserDoc === void 0 ? void 0 : (_existingUserDoc$enti = existingUserDoc.entity_analytics_monitoring) === null || _existingUserDoc$enti === void 0 ? void 0 : _existingUserDoc$enti.labels;

          // Merge API labels with existing labels, avoiding duplicates
          const mergedLabels = mergeApiLabels(existingLabels, apiLabels);
          await esClient.update({
            index,
            id: existingUserId,
            refresh: 'wait_for',
            doc: {
              ...user,
              '@timestamp': timestamp,
              event: {
                ...((_existingUserDoc$even = existingUserDoc === null || existingUserDoc === void 0 ? void 0 : existingUserDoc.event) !== null && _existingUserDoc$even !== void 0 ? _existingUserDoc$even : {}),
                ingested: timestamp,
                '@timestamp': timestamp
              },
              user: {
                ...user.user,
                is_privileged: true,
                entity: {
                  attributes: {
                    Privileged: true
                  }
                }
              },
              labels: {
                sources: updatedSources
              },
              entity_analytics_monitoring: {
                labels: mergedLabels
              }
            }
          });
          const updatedUser = await get(existingUserId);
          if (!updatedUser) {
            throw new Error(`Failed to retrieve updated user: ${existingUserId}`);
          }
          return updatedUser;
        }
      }
    }

    // Create new user if none exists
    // Check user count limit before creating new user
    const currentUserCount = await esClient.count({
      index,
      query: {
        term: {
          'user.is_privileged': true
        }
      }
    });
    if (currentUserCount.count >= maxUsersAllowed) {
      throw new Error(`Cannot create user: Maximum user limit of ${maxUsersAllowed} reached`);
    }

    // Prepare API labels with source 'api'
    const apiLabels = ((_user$entity_analytic2 = user.entity_analytics_monitoring) === null || _user$entity_analytic2 === void 0 ? void 0 : (_user$entity_analytic3 = _user$entity_analytic2.labels) === null || _user$entity_analytic3 === void 0 ? void 0 : _user$entity_analytic3.map(label => ({
      ...label,
      source: 'api'
    }))) || [];

    // Create new user
    const doc = (0, _lodash.merge)(user, {
      '@timestamp': timestamp,
      event: {
        ingested: timestamp,
        '@timestamp': timestamp
      },
      user: {
        is_privileged: true,
        entity: {
          attributes: {
            Privileged: true
          }
        }
      },
      labels: {
        sources: [source]
      },
      entity_analytics_monitoring: {
        labels: apiLabels
      }
    });
    const res = await esClient.index({
      index,
      refresh: 'wait_for',
      document: doc
    });
    const newUser = await get(res._id);
    if (!newUser) {
      throw new Error(`Failed to create user: ${res._id}`);
    }
    return newUser;
  };
  const get = async id => {
    const response = await esClient.get({
      index,
      id
    });
    return response.found ? {
      ...response._source,
      id: response._id
    } : undefined;
  };
  const update = async (id, user) => {
    var _existingUser$entity_, _user$entity_analytic4;
    // Get existing user to merge labels properly
    const existingUser = await get(id);
    if (!existingUser) {
      throw new Error(`User with id ${id} not found`);
    }
    const existingLabels = existingUser === null || existingUser === void 0 ? void 0 : (_existingUser$entity_ = existingUser.entity_analytics_monitoring) === null || _existingUser$entity_ === void 0 ? void 0 : _existingUser$entity_.labels;
    const apiLabels = (_user$entity_analytic4 = user.entity_analytics_monitoring) === null || _user$entity_analytic4 === void 0 ? void 0 : _user$entity_analytic4.labels;

    // Merge API labels with existing labels, avoiding duplicates
    const mergedLabels = mergeApiLabels(existingLabels, apiLabels);
    await esClient.update({
      index,
      refresh: 'wait_for',
      id,
      doc: {
        ...user,
        entity_analytics_monitoring: {
          labels: mergedLabels
        }
      }
    });
    return get(id);
  };
  const list = async kuery => {
    const query = kuery ? (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(kuery)) : {
      match_all: {}
    };
    const response = await esClient.search({
      size: 10000,
      index,
      query
    });
    return response.hits.hits.map(hit => ({
      id: hit._id,
      ...hit._source
    }));
  };
  const _delete = async id => {
    var _existingUser$labels;
    // Get existing user to check if it has non-API sources
    const existingUser = await get(id);
    if (!existingUser) {
      throw new Error(`User with id ${id} not found`);
    }
    const sources = (existingUser === null || existingUser === void 0 ? void 0 : (_existingUser$labels = existingUser.labels) === null || _existingUser$labels === void 0 ? void 0 : _existingUser$labels.sources) || [];
    const hasNonApiSources = sources.some(source => source !== 'api');
    if (hasNonApiSources) {
      var _existingUser$entity_2;
      // User has non-API sources, only remove API labels and API source
      const existingLabels = (existingUser === null || existingUser === void 0 ? void 0 : (_existingUser$entity_2 = existingUser.entity_analytics_monitoring) === null || _existingUser$entity_2 === void 0 ? void 0 : _existingUser$entity_2.labels) || [];
      const nonApiLabels = existingLabels.filter(label => label.source !== 'api');
      const nonApiSources = sources.filter(source => source !== 'api');
      await esClient.update({
        index,
        id,
        refresh: 'wait_for',
        doc: {
          labels: {
            sources: nonApiSources
          },
          entity_analytics_monitoring: {
            labels: nonApiLabels
          }
        }
      });
    } else {
      // User only has API source, delete the entire user
      await esClient.delete({
        index,
        id
      });
    }
  };
  return {
    create,
    get,
    update,
    list,
    delete: _delete
  };
};
exports.createPrivilegedUsersCrudService = createPrivilegedUsersCrudService;