"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildPrivilegedSearchBody = exports.applyPrivilegedUpdates = void 0;
var _lodash = require("lodash");
var _objectHash = _interopRequireDefault(require("object-hash"));
var _upsert = require("../../../bulk/upsert");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Build painless script for matchers
 * If we create new matcher types in the future, I think we may need to abandon
 * using painless due to script complexity and do the checking in TS code instead.
 * @param matchers - matcher objects containing fields and values to match against
 * @returns Painless script object
 */
const buildMatcherScript = matchers => {
  if (matchers.length === 0) {
    // if no matchers then everyone is privileged
    return {
      lang: 'painless',
      source: 'true'
    };
  }
  const source = matchers.map(valuesMatcherToPainless).join(' || ');
  return {
    lang: 'painless',
    source
  };
};

/**
 * Convert values matcher to painless script, this is currently the only supported matcher type.
 * A values matcher looks for specific values in specified fields like:
 * { fields: ['user.roles'], values: ['admin', 'superuser'] }
 */
const valuesMatcherToPainless = matcher => {
  const valuesLiteral = matcher.values.map(value => JSON.stringify(value)).join(', ');
  const fieldChecks = matcher.fields.map(field => `(doc.containsKey('${field}') && doc['${field}'].size() > 0 && [${valuesLiteral}].contains(doc['${field}'].value))`).join(' || ');
  return `(${fieldChecks})`;
};

/**
 * Extract unique matcher fields for _source includes
 */
const extractMatcherFields = matchers => {
  const fields = matchers.flatMap(matcher => matcher.fields);
  return Array.from(new Set(fields));
};

/**
 * Building privileged search body for matchers
 */
const buildPrivilegedSearchBody = (matchers, timeGte, afterKey, pageSize = 100) => {
  // this will get called multiple times with the same matchers during pagination
  const script = (0, _lodash.memoize)(buildMatcherScript, v => (0, _objectHash.default)(v))(matchers);
  return {
    size: 0,
    query: {
      range: {
        '@timestamp': {
          gte: timeGte,
          lte: 'now'
        }
      }
    },
    aggs: {
      privileged_user_status_since_last_run: {
        composite: {
          size: pageSize,
          sources: [{
            username: {
              terms: {
                field: 'user.name'
              }
            }
          }],
          ...(afterKey ? {
            after: afterKey
          } : {})
        },
        aggs: {
          latest_doc_for_user: {
            top_hits: {
              size: 1,
              sort: [{
                '@timestamp': {
                  order: 'desc'
                }
              }],
              script_fields: {
                'user.is_privileged': {
                  script
                }
              },
              _source: {
                includes: ['@timestamp', 'user.name', ...extractMatcherFields(matchers)]
              }
            }
          }
        }
      }
    }
  };
};
exports.buildPrivilegedSearchBody = buildPrivilegedSearchBody;
const applyPrivilegedUpdates = async ({
  dataClient,
  users,
  source
}) => {
  if (users.length === 0) return;
  const chunkSize = 500;
  const esClient = dataClient.deps.clusterClient.asCurrentUser;
  const opsForIntegration = (0, _upsert.makeIntegrationOpsBuilder)(dataClient);
  try {
    for (let start = 0; start < users.length; start += chunkSize) {
      const chunk = users.slice(start, start + chunkSize);
      const operations = opsForIntegration(chunk, source);
      if (operations.length > 0) {
        const resp = await esClient.bulk({
          refresh: 'wait_for',
          body: operations
        });
        const errors = (0, _utils.getErrorFromBulkResponse)(resp);
        dataClient.log('error', (0, _utils.errorsMsg)(errors));
      }
    }
  } catch (error) {
    dataClient.log('error', `Error applying privileged updates: ${error.message}`);
  }
};
exports.applyPrivilegedUpdates = applyPrivilegedUpdates;