"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.entityDetailsHighlightsServiceFactory = void 0;
var _findings_query_builders = require("@kbn/cloud-security-posture-common/utils/findings_query_builders");
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _lodash = require("lodash");
var _get_anonymized_values = require("@kbn/elastic-assistant-common/impl/data_anonymization/get_anonymized_values");
var _get_anonymized_data = require("@kbn/elastic-assistant-common/impl/data_anonymization/get_anonymized_data");
var _transform_raw_data = require("@kbn/elastic-assistant-common/impl/data_anonymization/transform_raw_data");
var _flatten_object = require("@kbn/object-utils/src/flatten_object");
var _get_risk_score = require("../risk_score/get_risk_score");
var _ml = require("../../../../common/utils/ml");
var _is_security_job = require("../../../../common/machine_learning/is_security_job");
var _constants = require("../../../../common/constants");
var _asset_criticality = require("../asset_criticality");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EMPTY_VULNERABILITIES_TOTAL = {
  [_findings_query_builders.VULNERABILITIES_RESULT_EVALUATION.NONE]: 0,
  [_findings_query_builders.VULNERABILITIES_RESULT_EVALUATION.CRITICAL]: 0,
  [_findings_query_builders.VULNERABILITIES_RESULT_EVALUATION.HIGH]: 0,
  [_findings_query_builders.VULNERABILITIES_RESULT_EVALUATION.MEDIUM]: 0,
  [_findings_query_builders.VULNERABILITIES_RESULT_EVALUATION.LOW]: 0
};
const entityDetailsHighlightsServiceFactory = ({
  logger,
  riskEngineClient,
  spaceId,
  esClient,
  assetCriticalityClient,
  soClient,
  uiSettingsClient,
  ml,
  anonymizationFields
}) => {
  let localReplacements = {};
  const localOnNewReplacements = newReplacements => {
    localReplacements = {
      ...localReplacements,
      ...newReplacements
    };
  };
  const fields = anonymizationFields.filter(fieldItem => fieldItem.allowed).map(fieldItem => ({
    field: fieldItem.field,
    include_unmapped: true
  }));
  return {
    async getRiskScoreData(entityType, entityIdentifier) {
      const engineStatus = await riskEngineClient.getStatus({
        namespace: spaceId
      });
      const getRiskScore = (0, _get_risk_score.createGetRiskScores)({
        logger,
        esClient,
        spaceId
      });
      let latestRiskScore = null;
      if (engineStatus.riskEngineStatus === 'ENABLED') {
        const riskScore = await getRiskScore({
          entityType: entityType,
          entityIdentifier,
          pagination: {
            querySize: 1,
            cursorStart: 0
          }
        });
        latestRiskScore = riskScore[0];
      }
      const anonymizedRiskScore = latestRiskScore ? [{
        score: [latestRiskScore.calculated_score_norm],
        id_field: [latestRiskScore.id_field],
        inputs: latestRiskScore.inputs.map(input => {
          var _input$risk_score$toS, _input$risk_score, _input$contribution_s, _input$contribution_s2, _input$description, _input$timestamp;
          return {
            risk_score: [(_input$risk_score$toS = (_input$risk_score = input.risk_score) === null || _input$risk_score === void 0 ? void 0 : _input$risk_score.toString()) !== null && _input$risk_score$toS !== void 0 ? _input$risk_score$toS : ''],
            contribution_score: [(_input$contribution_s = (_input$contribution_s2 = input.contribution_score) === null || _input$contribution_s2 === void 0 ? void 0 : _input$contribution_s2.toString()) !== null && _input$contribution_s !== void 0 ? _input$contribution_s : ''],
            description: [(_input$description = input.description) !== null && _input$description !== void 0 ? _input$description : ''],
            timestamp: [(_input$timestamp = input.timestamp) !== null && _input$timestamp !== void 0 ? _input$timestamp : '']
          };
        })
      }] : [];
      return anonymizedRiskScore;
    },
    async getAssetCriticalityData(entityField, entityIdentifier) {
      const param = {
        [entityField]: [entityIdentifier]
      };
      const criticalitiesQuery = (0, _asset_criticality.buildCriticalitiesQuery)(param);
      const criticalitySearchResponse = await assetCriticalityClient.search({
        query: criticalitiesQuery,
        size: 1,
        fields
      });
      const assetCriticalityAnonymized = criticalitySearchResponse.hits.hits.map(hit => (0, _transform_raw_data.transformRawDataToRecord)({
        anonymizationFields,
        currentReplacements: localReplacements,
        getAnonymizedValue: _elasticAssistantCommon.getAnonymizedValue,
        onNewReplacements: localOnNewReplacements,
        rawData: (0, _elasticAssistantCommon.getRawDataOrDefault)((0, _lodash.omit)(hit.fields, '_id')) // We need to exclude _id because asset criticality id contains user data
      }));
      return assetCriticalityAnonymized;
    },
    async getAnomaliesData(request, entityField, entityIdentifier, fromDate, toDate) {
      let anomaliesAnonymized = [];
      if (ml) {
        const jobs = await ml.jobServiceProvider(request, soClient).jobsSummary();
        const securityJobIds = jobs.filter(_is_security_job.isSecurityJob).map(j => j.id);
        const {
          getAnomaliesTableData
        } = ml.resultsServiceProvider(request, soClient);
        const anomalyScore = await uiSettingsClient.get(_constants.DEFAULT_ANOMALY_SCORE);
        const anomaliesData = await getAnomaliesTableData(securityJobIds, [{
          fieldName: entityField,
          fieldValue: entityIdentifier
        }], [], 'auto', [{
          min: (0, _ml.getThreshold)(anomalyScore, -1)
        }], fromDate, toDate, Intl.DateTimeFormat().resolvedOptions().timeZone, 500, 10, undefined);
        const jobNameById = jobs.reduce((acc, job) => {
          acc[job.id] = {
            name: job.customSettings.security_app_display_name,
            description: job.description
          };
          return acc;
        }, {});
        anomaliesAnonymized = anomaliesData.anomalies.map(anomaly => {
          // remove fields that could leak user data
          const formattedAnomaly = (0, _lodash.omit)(anomaly.source, ['partition_field_value', 'influencers', 'entityValue']);

          // the only ECS fields inside anomalies are entities data (user, host, ip)
          const relatedEntities = (0, _get_anonymized_data.getAnonymizedData)({
            anonymizationFields,
            currentReplacements: localReplacements,
            rawData: (0, _elasticAssistantCommon.getRawDataOrDefault)((0, _flatten_object.flattenObject)(formattedAnomaly)),
            getAnonymizedValue: _elasticAssistantCommon.getAnonymizedValue,
            getAnonymizedValues: _get_anonymized_values.getAnonymizedValues
          });
          localOnNewReplacements(relatedEntities.replacements);
          return (0, _flatten_object.flattenObject)({
            id: formattedAnomaly.job_id,
            score: formattedAnomaly.record_score,
            job: jobNameById[anomaly.jobId],
            entities: relatedEntities.anonymizedData
          });
        });
      }
      return anomaliesAnonymized;
    },
    async getVulnerabilityData(entityField, entityIdentifier) {
      var _vulnerabilities$aggr, _vulnerabilities$aggr2;
      const vulnerabilitiesQuery = (0, _findings_query_builders.getVulnerabilitiesQuery)({
        query: (0, _cloudSecurityPostureCommon.buildVulnerabilityEntityFlyoutPreviewQuery)(entityField, entityIdentifier),
        enabled: true,
        pageSize: 1,
        sort: [{
          'vulnerability.score.base': 'desc'
        }]
      });
      const vulnerabilities = entityField === 'host.name' // only hosts have vulnerabilities
      ? await esClient.search({
        ...vulnerabilitiesQuery,
        query: vulnerabilitiesQuery.query,
        _source: false,
        fields,
        size: 100
      }) : null;
      const vulnerabilitiesAggregations = vulnerabilities === null || vulnerabilities === void 0 ? void 0 : (_vulnerabilities$aggr = vulnerabilities.aggregations) === null || _vulnerabilities$aggr === void 0 ? void 0 : (_vulnerabilities$aggr2 = _vulnerabilities$aggr.count) === null || _vulnerabilities$aggr2 === void 0 ? void 0 : _vulnerabilities$aggr2.buckets;
      const vulnerabilitiesTotal = vulnerabilitiesAggregations ? Object.entries(vulnerabilitiesAggregations).reduce((acc, [key, value]) => {
        acc[key] = value.doc_count;
        return acc;
      }, EMPTY_VULNERABILITIES_TOTAL) : EMPTY_VULNERABILITIES_TOTAL;
      const vulnerabilitiesAnonymized = vulnerabilities === null || vulnerabilities === void 0 ? void 0 : vulnerabilities.hits.hits.map(hit => (0, _transform_raw_data.transformRawDataToRecord)({
        anonymizationFields,
        currentReplacements: localReplacements,
        getAnonymizedValue: _elasticAssistantCommon.getAnonymizedValue,
        onNewReplacements: localOnNewReplacements,
        rawData: (0, _elasticAssistantCommon.getRawDataOrDefault)(hit.fields)
      }));
      return {
        vulnerabilitiesAnonymized,
        vulnerabilitiesTotal
      };
    },
    getLocalReplacements(entityField, entityIdentifier) {
      // Ensure the entity identifier is present in the replacements
      const anonymizedEntityIdentifier = (0, _get_anonymized_data.getAnonymizedData)({
        anonymizationFields,
        currentReplacements: {},
        rawData: {
          [entityField]: [entityIdentifier]
        },
        getAnonymizedValue: _elasticAssistantCommon.getAnonymizedValue,
        getAnonymizedValues: _get_anonymized_values.getAnonymizedValues
      });
      localOnNewReplacements(anonymizedEntityIdentifier.replacements);
      return localReplacements;
    }
  };
};
exports.entityDetailsHighlightsServiceFactory = entityDetailsHighlightsServiceFactory;