"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkScheduleBackfill = void 0;
var _constants = require("@kbn/alerting-plugin/common/constants");
var _validations = require("../../../logic/bulk_actions/validations");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkScheduleBackfill = async ({
  rules,
  isDryRun,
  rulesClient,
  mlAuthz,
  runPayload
}) => {
  const errors = [];

  // In the first step, we validate if it is possible to schedule backfill for the rules
  const validatedRules = [];
  await Promise.all(rules.map(async rule => {
    try {
      await (0, _validations.validateBulkScheduleBackfill)({
        mlAuthz,
        rule
      });
      validatedRules.push(rule);
    } catch (error) {
      errors.push({
        item: rule,
        error
      });
    }
  }));
  if (isDryRun || validatedRules.length === 0) {
    return {
      backfilled: validatedRules,
      errors
    };
  }

  // Then if it's not a dry run, we schedule backfill for the rules that passed the validation
  const params = validatedRules.map(({
    id
  }) => ({
    ruleId: id,
    initiator: _constants.backfillInitiator.USER,
    ranges: [{
      start: runPayload.start_date,
      end: runPayload.end_date
    }]
  }));

  // Perform actual schedule using the rulesClient
  const backfillResults = await rulesClient.scheduleBackfill(params);
  const processedResults = (0, _utils.handleScheduleBackfillResults)({
    results: backfillResults,
    rules: validatedRules
  });
  return {
    backfilled: processedResults.backfilled,
    errors: [...errors, ...processedResults.errors]
  };
};
exports.bulkScheduleBackfill = bulkScheduleBackfill;