"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.replaceSignalsIndexAlias = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Updates aliases for the old and new concrete indexes specified, respectively
 * removing and adding them atomically.
 *
 * This is invoked as part of the finalization of a signals migration: once the
 * migrated index has been verified, its alias replaces the outdated index.
 *
 * @param esClient An {@link ElasticsearchClient}
 * @param alias name of the signals alias
 * @param newIndex name of the concrete signals index to be aliased
 * @param oldIndex name of the concrete signals index to be unaliased
 *
 * @throws if elasticsearch returns an error
 */
const replaceSignalsIndexAlias = async ({
  alias,
  esClient,
  newIndex,
  oldIndex,
  legacySiemSignalsAlias
}) => {
  await esClient.indices.updateAliases({
    actions: [{
      remove: {
        index: oldIndex,
        alias
      }
    }, {
      add: {
        index: newIndex,
        alias,
        is_write_index: false,
        is_hidden: true
      }
    }]
  });
  await esClient.indices.updateAliases({
    actions: [{
      remove: {
        index: oldIndex,
        alias: legacySiemSignalsAlias
      }
    }, {
      add: {
        index: newIndex,
        alias: legacySiemSignalsAlias,
        is_write_index: false
      }
    }]
  });
};
exports.replaceSignalsIndexAlias = replaceSignalsIndexAlias;