"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAssetInventoryInstallationStats = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _services = require("@kbn/fleet-plugin/server/services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Gets installation statistics for package policies with cloud connector support
 *
 * @param packagePolicies - List of package policies to process
 * @param agentPolicies - List of agent policies for enrichment
 * @returns Array of installation statistics
 */
const getInstalledPackagePolicies = (packagePolicies, agentPolicies) => {
  const installationStats = packagePolicies.flatMap(packagePolicy => packagePolicy.policy_ids.map(agentPolicyId => {
    var _packagePolicy$packag, _packagePolicy$packag2;
    const matchedAgentPolicy = agentPolicies === null || agentPolicies === void 0 ? void 0 : agentPolicies.find(agentPolicy => (agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.id) === agentPolicyId);
    const agentCounts = (matchedAgentPolicy === null || matchedAgentPolicy === void 0 ? void 0 : matchedAgentPolicy.agents) || 0;
    const isAgentless = !!(matchedAgentPolicy !== null && matchedAgentPolicy !== void 0 && matchedAgentPolicy.supports_agentless);
    return {
      package_policy_id: packagePolicy.id,
      package_name: ((_packagePolicy$packag = packagePolicy.package) === null || _packagePolicy$packag === void 0 ? void 0 : _packagePolicy$packag.name) || '',
      package_version: ((_packagePolicy$packag2 = packagePolicy.package) === null || _packagePolicy$packag2 === void 0 ? void 0 : _packagePolicy$packag2.version) || '',
      created_at: packagePolicy.created_at,
      agent_policy_id: agentPolicyId,
      agent_count: agentCounts,
      is_agentless: isAgentless,
      supports_cloud_connector: !!(packagePolicy !== null && packagePolicy !== void 0 && packagePolicy.supports_cloud_connector),
      cloud_connector_id: packagePolicy.cloud_connector_id || null
    };
  }));
  return installationStats;
};

/**
 * Collects installation statistics for Asset Inventory package policies
 *
 * This collector gathers information about:
 * - Package policies that support cloud connectors
 * - Agent policy associations
 * - Agent counts and agentless deployments
 * - Cloud connector references
 *
 * @param esClient - Elasticsearch client
 * @param soClient - Saved Objects client
 * @param coreServices - Core services promise
 * @param logger - Logger instance
 * @returns Promise resolving to array of installation statistics
 */
const getAssetInventoryInstallationStats = async (esClient, soClient, coreServices, logger) => {
  try {
    var _securitySolutionPlug;
    const [, securitySolutionPluginStartDeps] = await coreServices;
    const packagePolicyService = (_securitySolutionPlug = securitySolutionPluginStartDeps.fleet) === null || _securitySolutionPlug === void 0 ? void 0 : _securitySolutionPlug.packagePolicyService;
    if (!packagePolicyService) {
      logger.debug('Package policy service not available for Asset Inventory installation stats');
      return [];
    }

    // Query all package policies that support cloud connectors
    const packagePolicies = await packagePolicyService.list(soClient, {
      perPage: _common.SO_SEARCH_LIMIT,
      kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.supports_cloud_connector:true`
    });
    if (!packagePolicies || !packagePolicies.items.length) {
      logger.debug('No package policies with cloud connector support found for Asset Inventory');
      return [];
    }

    // Get all agent policies with agent counts for enrichment
    const agentPolicies = await _services.agentPolicyService.list(soClient, {
      perPage: _common.SO_SEARCH_LIMIT,
      kuery: '',
      esClient,
      withAgentCount: true
    });
    const installationStats = getInstalledPackagePolicies(packagePolicies.items, (agentPolicies === null || agentPolicies === void 0 ? void 0 : agentPolicies.items) || []);
    logger.info(`Collected Asset Inventory installation stats for ${installationStats.length} package policies`);
    return installationStats;
  } catch (error) {
    logger.error(`Failed to collect Asset Inventory installation stats: ${error.message}`);
    logger.error(error.stack);
    return [];
  }
};
exports.getAssetInventoryInstallationStats = getAssetInventoryInstallationStats;