"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAssetInventoryCloudConnectorUsageStats = void 0;
var _common = require("@kbn/fleet-plugin/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Type guard to check if vars is AwsCloudConnectorVars
 */
const isAwsVars = vars => {
  return typeof vars === 'object' && vars !== null && vars !== undefined && 'role_arn' in vars && 'external_id' in vars;
};

/**
 * Type guard to check if vars is AzureCloudConnectorVars
 */
const isAzureVars = vars => {
  return typeof vars === 'object' && vars !== null && vars !== undefined && 'tenant_id' in vars && 'client_id' in vars && 'azure_credentials_cloud_connector_id' in vars;
};

/**
 * Checks if the cloud connector has valid credentials based on cloud provider
 */
const hasValidCredentials = (cloudProvider, vars) => {
  switch (cloudProvider) {
    case 'aws':
      if (isAwsVars(vars)) {
        var _vars$role_arn, _vars$external_id;
        return !!((_vars$role_arn = vars.role_arn) !== null && _vars$role_arn !== void 0 && _vars$role_arn.value && (_vars$external_id = vars.external_id) !== null && _vars$external_id !== void 0 && _vars$external_id.value);
      }
      return false;
    case 'azure':
      if (isAzureVars(vars)) {
        var _vars$client_id, _vars$tenant_id, _vars$azure_credentia;
        return !!((_vars$client_id = vars.client_id) !== null && _vars$client_id !== void 0 && _vars$client_id.value && (_vars$tenant_id = vars.tenant_id) !== null && _vars$tenant_id !== void 0 && _vars$tenant_id.value && (_vars$azure_credentia = vars.azure_credentials_cloud_connector_id) !== null && _vars$azure_credentia !== void 0 && _vars$azure_credentia.value);
      }
      return false;
    default:
      return false;
  }
};
const getAssetInventoryCloudConnectorUsageStats = async (soClient, coreServices, logger) => {
  try {
    var _securitySolutionPlug, _securitySolutionPlug2;
    const [, securitySolutionPluginStartDeps] = await coreServices;

    // Get cloud connector service
    const cloudConnectorService = (_securitySolutionPlug = securitySolutionPluginStartDeps.fleet) === null || _securitySolutionPlug === void 0 ? void 0 : _securitySolutionPlug.cloudConnectorService;
    if (!cloudConnectorService) {
      logger.debug('Cloud connector service not available for Asset Inventory');
      return [];
    }

    // Get all cloud connectors
    const cloudConnectors = await cloudConnectorService.getList(soClient);
    if (!cloudConnectors.length) {
      logger.debug('No cloud connectors found for Asset Inventory');
      return [];
    }

    // Get package policies that use cloud connectors
    const packagePolicyService = (_securitySolutionPlug2 = securitySolutionPluginStartDeps.fleet) === null || _securitySolutionPlug2 === void 0 ? void 0 : _securitySolutionPlug2.packagePolicyService;
    if (!packagePolicyService) {
      logger.debug('Package policy service not available for Asset Inventory');
      return [];
    }
    const packagePolicies = await packagePolicyService.list(soClient, {
      perPage: _common.SO_SEARCH_LIMIT,
      kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.cloud_connector_id:*`
    });
    const stats = cloudConnectors.map(connector => {
      // Filter package policies for this cloud connector
      const connectorPackagePolicies = packagePolicies.items.filter(policy => policy.cloud_connector_id === connector.id);

      // Extract package policy IDs
      const packagePolicyIds = connectorPackagePolicies.map(policy => policy.id);
      return {
        id: connector.id,
        created_at: connector.created_at,
        updated_at: connector.updated_at,
        hasCredentials: hasValidCredentials(connector.cloudProvider, connector.vars),
        cloud_provider: connector.cloudProvider,
        packagePolicyIds,
        packagePolicyCount: connectorPackagePolicies.length
      };
    });
    logger.info(`Collected Asset Inventory cloud connector usage stats for ${stats.length} connectors`);
    return stats;
  } catch (error) {
    logger.error(`Failed to collect Asset Inventory cloud connector usage stats: ${error.message}`);
    return [];
  }
};
exports.getAssetInventoryCloudConnectorUsageStats = getAssetInventoryCloudConnectorUsageStats;