"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScriptsLibraryClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/files-plugin/server");
var _uuid = require("uuid");
var _assert = _interopRequireDefault(require("assert"));
var _constants = require("../../../../common/endpoint/constants");
var _scripts_library = require("../../lib/scripts_library");
var _common = require("./common");
var _utils = require("../../utils");
var _stringify = require("../../utils/stringify");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ScriptsLibraryClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "filesClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "username", void 0);
    this.logger = options.endpointService.createLogger('ScriptsLibraryClient');
    this.username = options.username;
    this.esClient = options.esClient || options.endpointService.getInternalEsClient();
    this.soClient = options.endpointService.savedObjects.createInternalScopedSoClient({
      spaceId: options.spaceId,
      readonly: false
    });
    this.filesClient = (0, _server.createEsFileClient)({
      metadataIndex: _scripts_library.SCRIPTS_LIBRARY_FILE_METADATA_INDEX_NAME,
      blobStorageIndex: _scripts_library.SCRIPTS_LIBRARY_FILE_DATA_INDEX_NAME,
      elasticsearchClient: this.esClient,
      logger: this.logger,
      maxSizeBytes: options.endpointService.getServerConfigValue('maxEndpointScriptFileSize')
    });
  }
  mapToSavedObjectProperties({
    name,
    platform,
    example,
    description,
    instructions,
    requiresInput,
    pathToExecutable
  }) {
    return {
      id: '',
      hash: '',
      name,
      platform,
      requires_input: requiresInput,
      description,
      instructions,
      example,
      pathToExecutable,
      created_by: '',
      updated_by: ''
    };
  }
  mapSoAttributesToEndpointScript({
    created_at: createdAt = '',
    updated_at: updatedAt = '',
    version = '',
    attributes: {
      id,
      name,
      platform,
      example,
      description,
      instructions,
      requires_input: requiresInput = false,
      pathToExecutable,
      created_by: createdBy,
      updated_by: updatedBy
    }
  }) {
    const downloadUri = _constants.SCRIPTS_LIBRARY_ITEM_DOWNLOAD_ROUTE.replace('{script_id}', id);
    return {
      id,
      name,
      platform: platform,
      downloadUri,
      requiresInput,
      description,
      instructions,
      example,
      pathToExecutable,
      createdBy,
      updatedBy,
      createdAt,
      updatedAt,
      version
    };
  }
  async create({
    file: _file,
    ...scriptDefinition
  }) {
    var _fileStream$hapi$file, _fileStream$hapi$head;
    const logger = this.logger.get('create');
    const scriptId = (0, _uuid.v4)();
    const fileStream = _file;
    const fileStorage = await this.filesClient.create({
      id: scriptId,
      metadata: {
        name: (_fileStream$hapi$file = fileStream.hapi.filename) !== null && _fileStream$hapi$file !== void 0 ? _fileStream$hapi$file : scriptDefinition.name.replace(/\D\W/g, '_'),
        mime: (_fileStream$hapi$head = fileStream.hapi.headers['content-type']) !== null && _fileStream$hapi$head !== void 0 ? _fileStream$hapi$head : 'application/octet-stream'
      }
    }).catch(error => {
      const message = `Unable to create File storage record: ${error.message}`;
      logger.error(message, {
        error
      });
      throw new _common.ScriptLibraryError(message, 500, error);
    });
    try {
      await fileStorage.uploadContent(fileStream, undefined, {
        transforms: [(0, _server.createFileHashTransform)()]
      });
      (0, _assert.default)(fileStorage.data.hash && fileStorage.data.hash.sha256, new _common.ScriptLibraryError('File hash was not generated after upload!'));
    } catch (error) {
      logger.error(`Error encountered while attempting to store file: ${error.message}`, {
        error
      });

      // attempt to delete the file record since we encountered an error during upload fo the file
      // Best effort being done here. If it fails, then just log the error since there is nothing else we can do.
      await fileStorage.delete().catch(deleteError => {
        logger.error(`Error encountered while attempting to cleanup file record: ${deleteError.message}`, {
          error: deleteError
        });
      });
      throw (0, _utils.wrapErrorIfNeeded)(error);
    }

    // Create the script entity in the saved objects store
    const soAttributes = this.mapToSavedObjectProperties(scriptDefinition);
    Object.assign(soAttributes, {
      id: scriptId,
      hash: fileStorage.data.hash.sha256,
      created_by: this.username,
      updated_by: this.username
    });
    try {
      logger.debug(() => `Creating script entity in saved objects store: ${(0, _stringify.stringify)(soAttributes)}`);
      const scriptSo = await this.soClient.create(_scripts_library.SCRIPTS_LIBRARY_SAVED_OBJECT_TYPE, soAttributes, {
        id: scriptId
      });
      logger.debug(() => `Script Created:\n${(0, _stringify.stringify)(scriptSo)}`);
      return this.mapSoAttributesToEndpointScript(scriptSo);
    } catch (error) {
      const message = `Attempt to create script record failed with: ${error.message}`;
      logger.error(message, {
        error
      });
      await fileStorage.delete().catch(deleteError => {
        logger.error(`Error encountered while attempting to cleanup file record: ${deleteError.message}`, {
          error: deleteError
        });
      });
      throw new _common.ScriptLibraryError(message, 500, error);
    }
  }
  async update(script) {
    throw new _common.ScriptLibraryError('Not implemented', 501);
  }
  async get(scriptId) {
    throw new _common.ScriptLibraryError('Not implemented', 501);
  }
  async list() {
    throw new _common.ScriptLibraryError('Not implemented', 501);
  }
  async delete(scriptId) {
    throw new _common.ScriptLibraryError('Not implemented', 501);
  }
  async download(scriptId) {
    throw new _common.ScriptLibraryError('Not implemented', 501);
  }
}
exports.ScriptsLibraryClient = ScriptsLibraryClient;