"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrowdstrikeActionsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/connector-schemas/crowdstrike/constants");
var _uuid = require("uuid");
var _crowdstrike = require("../../../../../../common/endpoint/service/response_actions/crowdstrike");
var _utils = require("./utils");
var _stringify = require("../../../../utils/stringify");
var _errors = require("../errors");
var _base_response_actions_client = require("../lib/base_response_actions_client");
var _utils2 = require("../../../../utils");
var _index_name_utilities = require("../../../../../../common/endpoint/utils/index_name_utilities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CrowdstrikeActionsClient extends _base_response_actions_client.ResponseActionsClientImpl {
  constructor({
    connectorActions,
    ...options
  }) {
    super(options);
    (0, _defineProperty2.default)(this, "agentType", 'crowdstrike');
    (0, _defineProperty2.default)(this, "connectorActionsClient", void 0);
    this.connectorActionsClient = connectorActions;
    connectorActions.setup(_constants.CONNECTOR_ID);
  }

  /**
   * Returns a list of all indexes for Crowdstrike data supported for response actions
   * @internal
   */
  async fetchIndexNames() {
    const cachedInfo = this.cache.get('fetchIndexNames');
    if (cachedInfo) {
      this.log.debug(`Returning cached response with list of index names:\n${(0, _stringify.stringify)(cachedInfo)}`);
      return cachedInfo;
    }
    const integrationNames = Object.keys(_crowdstrike.CROWDSTRIKE_INDEX_PATTERNS_BY_INTEGRATION);
    const fleetServices = this.options.endpointService.getInternalFleetServices(this.options.spaceId);
    const indexNamespaces = await fleetServices.getIntegrationNamespaces(integrationNames);
    const indexNames = [];
    for (const [integrationName, namespaces] of Object.entries(indexNamespaces)) {
      if (namespaces.length > 0) {
        const indexPatterns = _crowdstrike.CROWDSTRIKE_INDEX_PATTERNS_BY_INTEGRATION[integrationName];
        for (const indexPattern of indexPatterns) {
          indexNames.push(...namespaces.map(namespace => (0, _index_name_utilities.buildIndexNameWithNamespace)(indexPattern, namespace)));
        }
      }
    }
    this.cache.set('fetchIndexNames', indexNames);
    this.log.debug(() => `Crowdstrike indexes with namespace:\n${(0, _stringify.stringify)(indexNames)}`);
    return indexNames;
  }
  async fetchAgentPolicyInfo(agentIds) {
    const cacheKey = `fetchAgentPolicyInfo:${agentIds.sort().join('#')}`;
    const cacheResponse = this.cache.get(cacheKey);
    if (cacheResponse) {
      this.log.debug(() => `Cached agent policy info. found - returning it:\n${(0, _stringify.stringify)(cacheResponse)}`);
      return cacheResponse;
    }
    const esClient = this.options.esClient;
    const esSearchRequest = {
      index: await this.fetchIndexNames(),
      query: {
        bool: {
          filter: [{
            terms: {
              'device.id': agentIds
            }
          }]
        }
      },
      collapse: {
        field: 'device.id',
        inner_hits: {
          name: 'most_recent',
          size: 1,
          _source: ['agent', 'device.id', '@timestamp'],
          sort: [{
            '@timestamp': 'desc'
          }]
        }
      },
      _source: false,
      ignore_unavailable: true
    };
    if (!esSearchRequest.index || esSearchRequest.index.length === 0) {
      throw new _errors.ResponseActionsClientError(`Unable to build list of indexes while retrieving policy information for Crowdstrike agents [${agentIds.join(', ')}]. Check to ensure at least one integration policy exists.`, 400);
    }
    this.log.debug(() => `Searching for agents with:\n${(0, _stringify.stringify)(esSearchRequest)}`);

    // Get the latest ingested document for each agent ID
    const crowdstrikeEsResults = await esClient.search(esSearchRequest).catch(_utils2.catchAndWrapError);
    this.log.debug(() => `Records found:\n${(0, _stringify.stringify)(crowdstrikeEsResults, 20)}`);
    const agentIdsFound = [];
    const fleetAgentIdToCrowdstrikeAgentIdMap = crowdstrikeEsResults.hits.hits.reduce((acc, esDoc) => {
      var _esDoc$inner_hits;
      const doc = (_esDoc$inner_hits = esDoc.inner_hits) === null || _esDoc$inner_hits === void 0 ? void 0 : _esDoc$inner_hits.most_recent.hits.hits[0]._source;
      if (doc) {
        agentIdsFound.push(doc.device.id);
        acc[doc.agent.id] = doc.device.id;
      }
      return acc;
    }, {});
    const elasticAgentIds = Object.keys(fleetAgentIdToCrowdstrikeAgentIdMap);
    if (elasticAgentIds.length === 0) {
      throw new _errors.ResponseActionsClientError(`Unable to find elastic agent IDs for Crowdstrike agent ids: [${agentIds.join(', ')}]`, 400);
    }

    // ensure all agent ids were found
    for (const agentId of agentIds) {
      if (!agentIdsFound.includes(agentId)) {
        throw new _errors.ResponseActionsClientError(`Crowdstrike agent id [${agentId}] not found`, 404);
      }
    }
    const agentPolicyInfo = await this.fetchFleetInfoForAgents(elasticAgentIds);
    for (const agentInfo of agentPolicyInfo) {
      agentInfo.agentId = fleetAgentIdToCrowdstrikeAgentIdMap[agentInfo.elasticAgentId];
    }
    this.cache.set(cacheKey, agentPolicyInfo);
    return agentPolicyInfo;
  }
  async writeActionRequestToEndpointIndex(actionRequest) {
    var _actionRequest$meta;
    const agentId = actionRequest.endpoint_ids[0];
    const hostname = await this.getHostNameByAgentId(agentId);
    return super.writeActionRequestToEndpointIndex({
      ...actionRequest,
      hosts: {
        [agentId]: {
          name: hostname
        }
      },
      meta: {
        hostName: hostname,
        ...((_actionRequest$meta = actionRequest.meta) !== null && _actionRequest$meta !== void 0 ? _actionRequest$meta : {})
      }
    });
  }

  /**
   * Sends actions to Crowdstrike directly (via Connector)
   * @internal
   */
  async sendAction(actionType, actionParams) {
    const executeOptions = {
      params: {
        subAction: actionType,
        subActionParams: actionParams
      }
    };
    this.log.debug(() => `calling connector actions 'execute()' for Crowdstrike with:\n${(0, _stringify.stringify)(executeOptions)}`);
    const actionSendResponse = await this.connectorActionsClient.execute(executeOptions);
    if (actionSendResponse.status === 'error') {
      this.log.error((0, _stringify.stringify)(actionSendResponse));
      throw new _errors.ResponseActionsClientError(`Attempt to send [${actionType}] to Crowdstrike failed: ${actionSendResponse.serviceMessage || actionSendResponse.message}`, 500, actionSendResponse);
    } else {
      this.log.debug(() => `Response:\n${(0, _stringify.stringify)(actionSendResponse)}`);
    }
    return actionSendResponse;
  }
  async getHostNameByAgentId(agentId) {
    const search = {
      // Multiple indexes:  .falcon, .fdr, .host, .alert
      index: ['logs-crowdstrike*'],
      size: 1,
      _source: ['host.hostname', 'host.name'],
      query: {
        bool: {
          filter: [{
            term: {
              'device.id': agentId
            }
          }]
        }
      }
    };
    try {
      var _result$hits$hits, _result$hits$hits$, _result$hits$hits$$_s;
      const result = await this.options.esClient.search(search, {
        ignore: [404]
      });

      // Check if host name exists
      const host = (_result$hits$hits = result.hits.hits) === null || _result$hits$hits === void 0 ? void 0 : (_result$hits$hits$ = _result$hits$hits[0]) === null || _result$hits$hits$ === void 0 ? void 0 : (_result$hits$hits$$_s = _result$hits$hits$._source) === null || _result$hits$hits$$_s === void 0 ? void 0 : _result$hits$hits$$_s.host;
      const hostName = (host === null || host === void 0 ? void 0 : host.name) || (host === null || host === void 0 ? void 0 : host.hostname);
      if (!hostName) {
        throw new _errors.ResponseActionsClientError(`Host name not found in the event document for agentId: ${agentId}`, 404);
      }
      return hostName;
    } catch (err) {
      var _err$statusCode;
      throw new _errors.ResponseActionsClientError(`Failed to fetch event document: ${err.message}`, (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 500, err);
    }
  }
  async validateRequest(
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  payload) {
    // TODO:PT support multiple agents
    if (payload.endpoint_ids.length > 1) {
      return {
        isValid: false,
        error: new _errors.ResponseActionsClientError(`[body.endpoint_ids]: Multiple agents IDs not currently supported for Crowdstrike`, 400)
      };
    }
    return super.validateRequest(payload);
  }
  async isolate(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: undefined,
      command: 'isolate'
    };
    let actionResponse;
    if (!reqIndexOptions.error) {
      var _error;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        if (!reqIndexOptions.actionId) {
          reqIndexOptions.actionId = (0, _uuid.v4)();
        }
        try {
          actionResponse = await this.sendAction(_constants.SUB_ACTION.HOST_ACTIONS, {
            ids: actionRequest.endpoint_ids,
            actionParameters: {
              comment: this.buildExternalComment(reqIndexOptions)
            },
            command: 'contain'
          });
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error = error) === null || _error === void 0 ? void 0 : _error.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const actionRequestDoc = await this.writeActionRequestToEndpointIndex(reqIndexOptions);

    // Ensure actionResponse is assigned before using it
    if (actionResponse) {
      await this.completeCrowdstrikeAction(actionResponse, actionRequestDoc);
    }
    await this.updateCases({
      command: reqIndexOptions.command,
      caseIds: reqIndexOptions.case_ids,
      alertIds: reqIndexOptions.alert_ids,
      actionId: actionRequestDoc.EndpointActions.action_id,
      hosts: actionRequest.endpoint_ids.map(agentId => {
        var _actionRequestDoc$End, _actionRequestDoc$End2;
        return {
          hostId: agentId,
          hostname: (_actionRequestDoc$End = (_actionRequestDoc$End2 = actionRequestDoc.EndpointActions.data.hosts) === null || _actionRequestDoc$End2 === void 0 ? void 0 : _actionRequestDoc$End2[agentId].name) !== null && _actionRequestDoc$End !== void 0 ? _actionRequestDoc$End : ''
        };
      }),
      comment: reqIndexOptions.comment
    });
    return this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id);
  }
  async release(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: undefined,
      command: 'unisolate'
    };
    let actionResponse;
    if (!reqIndexOptions.error) {
      var _error2;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          actionResponse = await this.sendAction(_constants.SUB_ACTION.HOST_ACTIONS, {
            ids: actionRequest.endpoint_ids,
            command: 'lift_containment',
            comment: this.buildExternalComment(reqIndexOptions)
          });
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error2 = error) === null || _error2 === void 0 ? void 0 : _error2.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const actionRequestDoc = await this.writeActionRequestToEndpointIndex(reqIndexOptions);

    // Ensure actionResponse is assigned before using it
    if (actionResponse) {
      await this.completeCrowdstrikeAction(actionResponse, actionRequestDoc);
    }
    await this.updateCases({
      command: reqIndexOptions.command,
      caseIds: reqIndexOptions.case_ids,
      alertIds: reqIndexOptions.alert_ids,
      actionId: actionRequestDoc.EndpointActions.action_id,
      hosts: actionRequest.endpoint_ids.map(agentId => {
        var _actionRequestDoc$End3, _actionRequestDoc$End4;
        return {
          hostId: agentId,
          hostname: (_actionRequestDoc$End3 = (_actionRequestDoc$End4 = actionRequestDoc.EndpointActions.data.hosts) === null || _actionRequestDoc$End4 === void 0 ? void 0 : _actionRequestDoc$End4[agentId].name) !== null && _actionRequestDoc$End3 !== void 0 ? _actionRequestDoc$End3 : ''
        };
      }),
      comment: reqIndexOptions.comment
    });
    return this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id);
  }
  async runscript(actionRequest, options) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'runscript'
    };
    let actionResponse;
    if (!reqIndexOptions.error) {
      var _error3;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        if (!reqIndexOptions.actionId) {
          reqIndexOptions.actionId = (0, _uuid.v4)();
        }
        try {
          actionResponse = await this.sendAction(_constants.SUB_ACTION.EXECUTE_ADMIN_RTR, {
            actionParameters: {
              comment: this.buildExternalComment(reqIndexOptions)
            },
            command: (0, _utils.mapParametersToCrowdStrikeArguments)('runscript', actionRequest.parameters),
            endpoint_ids: actionRequest.endpoint_ids
          });
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error3 = error) === null || _error3 === void 0 ? void 0 : _error3.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const actionRequestDoc = await this.writeActionRequestToEndpointIndex(reqIndexOptions);

    // Ensure actionResponse is assigned before using it
    if (actionResponse) {
      await this.completeCrowdstrikeBatchAction(actionResponse, actionRequestDoc);
    }
    await this.updateCases({
      command: reqIndexOptions.command,
      caseIds: reqIndexOptions.case_ids,
      alertIds: reqIndexOptions.alert_ids,
      actionId: actionRequestDoc.EndpointActions.action_id,
      hosts: actionRequest.endpoint_ids.map(agentId => {
        var _actionRequestDoc$End5, _actionRequestDoc$End6;
        return {
          hostId: agentId,
          hostname: (_actionRequestDoc$End5 = (_actionRequestDoc$End6 = actionRequestDoc.EndpointActions.data.hosts) === null || _actionRequestDoc$End6 === void 0 ? void 0 : _actionRequestDoc$End6[agentId].name) !== null && _actionRequestDoc$End5 !== void 0 ? _actionRequestDoc$End5 : ''
        };
      }),
      comment: reqIndexOptions.comment
    });
    return this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id);
  }
  async completeCrowdstrikeBatchAction(actionResponse, doc) {
    var _actionResponse$data, _actionResponse$data2, _actionResponse$data3, _actionResponse$data4;
    const agentId = doc.agent.id;
    const stdout = ((_actionResponse$data = actionResponse.data) === null || _actionResponse$data === void 0 ? void 0 : _actionResponse$data.combined.resources[agentId].stdout) || '';
    const stderr = ((_actionResponse$data2 = actionResponse.data) === null || _actionResponse$data2 === void 0 ? void 0 : _actionResponse$data2.combined.resources[agentId].stderr) || '';
    const error = (_actionResponse$data3 = actionResponse.data) === null || _actionResponse$data3 === void 0 ? void 0 : (_actionResponse$data4 = _actionResponse$data3.combined.resources[agentId].errors) === null || _actionResponse$data4 === void 0 ? void 0 : _actionResponse$data4[0];
    const options = {
      actionId: doc.EndpointActions.action_id,
      agentId,
      data: {
        ...doc.EndpointActions.data,
        output: {
          content: {
            stdout,
            stderr,
            code: '200'
          },
          type: 'text'
        }
      },
      ...(error ? {
        error: {
          code: error.code,
          message: `Crowdstrike action failed: ${error.message}`
        }
      } : {})
    };
    const responseDoc = await this.writeActionResponseToEndpointIndex(options);
    // telemetry event for completed action
    await this.sendActionResponseTelemetry([responseDoc]);
  }
  async completeCrowdstrikeAction(actionResponse, doc) {
    var _actionResponse$data5, _actionResponse$data6;
    const options = {
      actionId: doc.EndpointActions.action_id,
      agentId: doc.agent.id,
      data: doc.EndpointActions.data,
      ...(actionResponse !== null && actionResponse !== void 0 && (_actionResponse$data5 = actionResponse.data) !== null && _actionResponse$data5 !== void 0 && (_actionResponse$data6 = _actionResponse$data5.errors) !== null && _actionResponse$data6 !== void 0 && _actionResponse$data6.length ? {
        error: {
          code: '500',
          message: `Crowdstrike action failed: ${actionResponse.data.errors[0].message}`
        }
      } : {})
    };
    const responseDoc = await this.writeActionResponseToEndpointIndex(options);
    // telemetry event for completed action
    await this.sendActionResponseTelemetry([responseDoc]);
  }
  async getCustomScripts() {
    try {
      var _customScriptsRespons;
      const customScriptsResponse = await this.sendAction(_constants.SUB_ACTION.GET_RTR_CLOUD_SCRIPTS, {});
      const resources = ((_customScriptsRespons = customScriptsResponse.data) === null || _customScriptsRespons === void 0 ? void 0 : _customScriptsRespons.resources) || [];
      // Transform CrowdStrike script resources to ResponseActionScriptsApiResponse format
      const data = resources.map(script => ({
        // due to External EDR's schema nature - we expect a maybe() everywhere - empty strings are needed
        id: script.id || '',
        name: script.name || '',
        description: script.description || ''
      }));
      return {
        data
      };
    } catch (err) {
      const error = new _errors.ResponseActionsClientError(`Failed to fetch Crowdstrike scripts, failed with: ${err.message}`, 500, err);
      this.log.error(error);
      throw error;
    }
  }
  async processPendingActions({
    abortSignal,
    addToQueue
  }) {
    // TODO:PT implement resolving of pending crowdstrike actions
    // if (abortSignal.aborted) {
    //   return;
    // }
  }
}
exports.CrowdstrikeActionsClient = CrowdstrikeActionsClient;