"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TelemetryConfigWatcher = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("@kbn/fleet-plugin/common");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _common2 = require("@kbn/spaces-plugin/common");
var _policy = require("../../../../common/endpoint/service/policy");
var _stringify = require("../../utils/stringify");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class TelemetryConfigWatcher {
  constructor(policyService, esStart, endpointAppContextService, options = {
    immediateRetry: false
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "policyService", void 0);
    (0, _defineProperty2.default)(this, "endpointAppContextService", void 0);
    (0, _defineProperty2.default)(this, "subscription", void 0);
    (0, _defineProperty2.default)(this, "retryOptions", void 0);
    this.policyService = policyService;
    this.esClient = esStart.client.asInternalUser;
    this.endpointAppContextService = endpointAppContextService;
    this.logger = endpointAppContextService.createLogger(this.constructor.name);
    this.retryOptions = {
      retries: 4,
      minTimeout: options.immediateRetry ? 0 : 1000
    };
  }
  start(telemetryConfigProvider) {
    var _telemetryConfigProvi;
    this.subscription = (_telemetryConfigProvi = telemetryConfigProvider.getObservable()) === null || _telemetryConfigProvi === void 0 ? void 0 : _telemetryConfigProvi.subscribe(this.watch.bind(this));
  }
  stop() {
    if (this.subscription) {
      this.subscription.unsubscribe();
    }
  }
  async watch(isTelemetryEnabled) {
    let page = 1;
    let response;
    let updated = 0;
    let failed = 0;
    let conflicts = 0;
    this.logger.debug(`Checking Endpoint policies to update due to changed global telemetry config setting. (New value: ${isTelemetryEnabled})`);
    do {
      try {
        response = await (0, _pRetry.default)(attemptCount => {
          const soClient = this.endpointAppContextService.savedObjects.createInternalUnscopedSoClient(false);
          return this.policyService.list(soClient, {
            page,
            perPage: 100,
            kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: endpoint`,
            spaceId: '*'
          }).then(result => {
            this.logger.debug(`Retrieved page [${page}] of endpoint package policies on attempt [${attemptCount}]`);
            return result;
          });
        }, {
          onFailedAttempt: error => this.logger.debug(() => `Failed to fetch list of package policies. Attempt [${error.attemptNumber}] for page [${page}] returned error: ${error.name}, ${error.message}`),
          ...this.retryOptions
        });
      } catch (e) {
        this.logger.warn(`Unable to verify endpoint policies in line with telemetry change: failed to fetch package policies: ${(0, _stringify.stringify)(e)}`);
        return;
      }
      this.logger.debug(() => `Processing page [${response.page}] with [${response.items.length}] policy(s)`);
      const updatesBySpace = {};
      for (const policy of response.items) {
        const updatePolicy = (0, _policy.getPolicyDataForUpdate)(policy);
        const policyConfig = updatePolicy.inputs[0].config.policy.value;
        if (isTelemetryEnabled !== policyConfig.global_telemetry_enabled) {
          var _policy$spaceIds$at, _policy$spaceIds;
          this.logger.debug(`Endpoint policy [${policy.id}] needs update to global telemetry enabled setting (currently set to [${policyConfig.global_telemetry_enabled}])`);
          const policySpace = (_policy$spaceIds$at = (_policy$spaceIds = policy.spaceIds) === null || _policy$spaceIds === void 0 ? void 0 : _policy$spaceIds.at(0)) !== null && _policy$spaceIds$at !== void 0 ? _policy$spaceIds$at : _common2.DEFAULT_SPACE_ID;
          policyConfig.global_telemetry_enabled = isTelemetryEnabled;
          if (!updatesBySpace[policySpace]) {
            updatesBySpace[policySpace] = [];
          }
          updatesBySpace[policySpace].push({
            ...updatePolicy,
            id: policy.id
          });
        }
      }
      for (const [spaceId, spaceUpdates] of Object.entries(updatesBySpace)) {
        this.logger.debug(`Updating [${spaceUpdates.length}] policies for space [${spaceId}]`);
        const soClientForSpace = this.endpointAppContextService.savedObjects.createInternalScopedSoClient({
          spaceId,
          readonly: false
        });
        try {
          var _updateResult$updated, _updateResult$updated2;
          const updateResult = await (0, _pRetry.default)(() => this.policyService.bulkUpdate(soClientForSpace, this.esClient, spaceUpdates), {
            onFailedAttempt: error => this.logger.debug(`Failed to bulk update package policies on ${error.attemptNumber}. attempt, reason: ${(0, _stringify.stringify)(error)}`),
            ...this.retryOptions
          });

          // Conflicts are expected on Serverless environments with multiple Kibana instances trying to do the same update, hence it's debug only.
          const failedPolicies = updateResult.failedPolicies.filter(entry => !('statusCode' in entry.error && entry.error.statusCode === 409));
          if (failedPolicies.length) {
            this.logger.warn(`Cannot update telemetry flag in the following policies:\n${failedPolicies.map(entry => `- id: ${entry.packagePolicy.id}, error: ${(0, _stringify.stringify)(entry.error)}`).join('\n')}`);
          }
          const conflictedPolicies = updateResult.failedPolicies.filter(entry => 'statusCode' in entry.error && entry.error.statusCode === 409);
          if (conflictedPolicies.length) {
            this.logger.debug(`Following policies are conflicted:\n${conflictedPolicies.map(entry => `- id: ${entry.packagePolicy.id}, error: ${(0, _stringify.stringify)(entry.error)}`).join('\n')}`);
          }
          updated += (_updateResult$updated = (_updateResult$updated2 = updateResult.updatedPolicies) === null || _updateResult$updated2 === void 0 ? void 0 : _updateResult$updated2.length) !== null && _updateResult$updated !== void 0 ? _updateResult$updated : 0;
          failed += failedPolicies.length;
          conflicts += conflictedPolicies.length;
        } catch (e) {
          this.logger.warn(`Unable to update telemetry config state to ${isTelemetryEnabled} in space [${spaceId}] for policies: ${spaceUpdates.map(update => update.id)}\n\n${(0, _stringify.stringify)(e)}`);
          failed += spaceUpdates.length;
        }
      }
      page++;
    } while (response.page * response.perPage < response.total);
    if (updated > 0 || failed > 0) {
      this.logger.info(`Finished updating global_telemetry_enabled flag to ${isTelemetryEnabled} in Defend package policies: ${updated} succeeded, ${failed} failed, ${conflicts} conflicts.`);
    } else {
      this.logger.debug(`Done checking Endpoint policies due to changed global telemetry config setting. (New value: ${isTelemetryEnabled})`);
    }
  }
}
exports.TelemetryConfigWatcher = TelemetryConfigWatcher;