"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSourcererDataView = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _sourcerer = require("../../../common/utils/sourcerer");
var _constants = require("../../../common/constants");
var _translations = require("../../data_view_manager/components/data_view_picker/translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates the default data view used by security solution.
 * If the data view already exists, it will update the pattern list if it is different
 * or the name if it is incorrect.
 */
const getDefaultDataView = async ({
  dataViewService,
  allDataViews,
  dataViewId,
  patternListFormatted,
  patternListAsTitle
}) => {
  let defaultDataView;
  let dataView;
  const siemDataViewExist = allDataViews.find(dv => dv.id === dataViewId);
  if (siemDataViewExist === undefined) {
    var _dataView$id;
    try {
      dataView = await dataViewService.createAndSave({
        allowNoIndex: true,
        id: dataViewId,
        title: patternListAsTitle,
        timeFieldName: _constants.DEFAULT_TIME_FIELD,
        name: _translations.DEFAULT_SECURITY_DATA_VIEW
      },
      // Override property - if a data view exists with the security solution pattern
      // delete it and replace it with our data view
      true);
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      if (err.name === 'DuplicateDataViewError' || error.statusCode === 409) {
        dataView = await dataViewService.get(dataViewId);
      } else {
        throw error;
      }
    }
    defaultDataView = {
      id: (_dataView$id = dataView.id) !== null && _dataView$id !== void 0 ? _dataView$id : dataViewId,
      patternList: dataView.title.split(','),
      title: dataView.title
    };
  } else {
    let patterns = (0, _sourcerer.ensurePatternFormat)(siemDataViewExist.title.split(','));
    const siemDataViewTitle = siemDataViewExist ? patterns.join() : '';
    const arePatternsDifferent = patternListAsTitle !== siemDataViewTitle;
    const isDefaultDataViewName = siemDataViewExist.name === _translations.DEFAULT_SECURITY_DATA_VIEW;

    // Update the saved object if the pattern list is different or the name is incorrect
    if (arePatternsDifferent || !isDefaultDataViewName) {
      dataView = await dataViewService.get(dataViewId);
      if (arePatternsDifferent) {
        patterns = patternListFormatted;
        dataView.title = patternListAsTitle;
      }
      if (!isDefaultDataViewName) {
        dataView.name = _translations.DEFAULT_SECURITY_DATA_VIEW;
      }
      await dataViewService.updateSavedObject(dataView);
    }
    defaultDataView = {
      id: dataViewId,
      patternList: patterns,
      title: patternListAsTitle
    };
  }
  const existingPatternList = await dataViewService.getExistingIndices(defaultDataView.patternList);
  defaultDataView = {
    ...defaultDataView,
    patternList: existingPatternList
  };
  return defaultDataView;
};

/**
 * Creates the alert data view used by security solution.
 * If the data view already exists, it will update the name if it is incorrect.
 */
const getAlertDataView = async ({
  dataViewService,
  allDataViews,
  alertDetails: {
    dataViewId,
    indexName
  }
}) => {
  let alertDataView;
  let dataView;
  const dataViewExist = allDataViews.find(dv => dv.id === dataViewId);
  if (indexName && dataViewId && dataViewExist === undefined) {
    var _dataView$id2;
    try {
      dataView = await dataViewService.createAndSave({
        allowNoIndex: true,
        id: dataViewId,
        title: indexName,
        timeFieldName: _constants.DEFAULT_TIME_FIELD,
        name: _translations.DEFAULT_SECURITY_ALERT_DATA_VIEW,
        managed: true
      },
      // Override property - if a data view exists with the security solution pattern
      // delete it and replace it with our data view
      true);
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      if (err.name === 'DuplicateDataViewError' || error.statusCode === 409) {
        dataView = await dataViewService.get(dataViewId);
      } else {
        throw error;
      }
    }
    alertDataView = {
      id: (_dataView$id2 = dataView.id) !== null && _dataView$id2 !== void 0 ? _dataView$id2 : dataViewId,
      patternList: dataView.title.split(','),
      title: dataView.title
    };
  } else {
    // Update the saved object if the name is incorrect
    if (dataViewId && (dataViewExist === null || dataViewExist === void 0 ? void 0 : dataViewExist.name) !== _translations.DEFAULT_SECURITY_ALERT_DATA_VIEW) {
      const dv = await dataViewService.get(dataViewId);
      dv.name = _translations.DEFAULT_SECURITY_ALERT_DATA_VIEW;
      await dataViewService.updateSavedObject(dv);
    }
    alertDataView = {
      id: dataViewId !== null && dataViewId !== void 0 ? dataViewId : '',
      patternList: indexName ? [indexName] : [],
      title: indexName !== null && indexName !== void 0 ? indexName : ''
    };
  }
  return alertDataView;
};

/**
 * Creates the attack data view used by security solution.
 */
const getAttackDataView = async ({
  dataViewService,
  allDataViews,
  attackDetails: {
    dataViewId,
    patternList
  }
}) => {
  let attackDataView;
  let dataView;
  const dataViewExist = allDataViews.find(dv => dv.id === dataViewId);
  if (patternList && dataViewId && dataViewExist === undefined) {
    var _dataView$id3;
    const patternListFormatted = (0, _sourcerer.ensurePatternFormat)(patternList);
    const patternListAsTitle = patternListFormatted.join();
    try {
      dataView = await dataViewService.createAndSave({
        allowNoIndex: true,
        id: dataViewId,
        title: patternListAsTitle,
        timeFieldName: _constants.DEFAULT_TIME_FIELD,
        name: _translations.DEFAULT_SECURITY_ATTACK_DATA_VIEW,
        managed: true
      },
      // Override property - if a data view exists with the security solution pattern
      // delete it and replace it with our data view
      true);
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      if (err.name === 'DuplicateDataViewError' || error.statusCode === 409) {
        dataView = await dataViewService.get(dataViewId);
      } else {
        throw error;
      }
    }
    attackDataView = {
      id: (_dataView$id3 = dataView.id) !== null && _dataView$id3 !== void 0 ? _dataView$id3 : dataViewId,
      patternList: dataView.title.split(','),
      title: dataView.title
    };
  } else {
    attackDataView = {
      id: dataViewId !== null && dataViewId !== void 0 ? dataViewId : '',
      patternList: patternList ? patternList : [],
      title: patternList ? patternList.join() : ''
    };
  }
  return attackDataView;
};
/**
 * Returns sourcerer data view used by security solution.
 * Creates the default, alert, and attack data views if they do not exist.
 */
const createSourcererDataView = async ({
  dataViewService,
  defaultDetails: {
    dataViewId,
    patternList
  },
  alertDetails,
  attackDetails
}) => {
  if (dataViewId === null) {
    return;
  }
  let allDataViews = await dataViewService.getIdsWithTitle();
  const patternListFormatted = (0, _sourcerer.ensurePatternFormat)(patternList);
  const patternListAsTitle = patternListFormatted.join();
  const defaultDataView = await getDefaultDataView({
    dataViewService,
    allDataViews,
    dataViewId,
    patternListFormatted,
    patternListAsTitle
  });
  const alertDataView = await getAlertDataView({
    dataViewService,
    allDataViews,
    alertDetails
  });
  let attackDataView;
  if (attackDetails) {
    attackDataView = await getAttackDataView({
      dataViewService,
      allDataViews,
      attackDetails
    });
  }
  if (allDataViews.some(dv => dv.id === dataViewId)) {
    allDataViews = allDataViews.map(v => v.id === dataViewId ? {
      ...v,
      title: patternListAsTitle
    } : v);
  } else if (defaultDataView !== null) {
    var _defaultDataView$id;
    allDataViews.push({
      id: (_defaultDataView$id = defaultDataView.id) !== null && _defaultDataView$id !== void 0 ? _defaultDataView$id : dataViewId,
      title: defaultDataView === null || defaultDataView === void 0 ? void 0 : defaultDataView.title
    });
  }
  return {
    defaultDataView,
    kibanaDataViews: allDataViews.map(dv => dv.id === dataViewId ? defaultDataView : {
      id: dv.id,
      patternList: dv.title.split(','),
      title: dv.title
    }),
    alertDataView,
    ...(attackDataView && {
      attackDataView
    })
  };
};
exports.createSourcererDataView = createSourcererDataView;