"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemDashboardMigrationsService = exports.CREATE_MIGRATION_BODY_BATCH_SIZE = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _experimental_features_service = require("../../../common/experimental_features_service");
var _use_license = require("../../../common/hooks/use_license");
var i18n = _interopRequireWildcard(require("./translations"));
var api = _interopRequireWildcard(require("../api"));
var _missing_capabilities_notification = require("../../common/service/notifications/missing_capabilities_notification");
var _no_connector_notification = require("../../common/service/notifications/no_connector_notification");
var _constants = require("../../../../common/siem_migrations/constants");
var _success_notification = require("./notification/success_notification");
var _capabilities = require("../../common/service/capabilities");
var _capabilities2 = require("./capabilities");
var _service = require("../../common/service");
var _constants2 = require("../../common/constants");
var _telemetry = require("./telemetry");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CREATE_MIGRATION_BODY_BATCH_SIZE = exports.CREATE_MIGRATION_BODY_BATCH_SIZE = 50;
class SiemDashboardMigrationsService extends _service.SiemMigrationsServiceBase {
  constructor(core, plugins, telemetryService) {
    super(core, plugins);
    (0, _defineProperty2.default)(this, "telemetry", void 0);
    this.telemetry = new _telemetry.SiemDashboardMigrationsTelemetry(telemetryService);
  }

  /** Accessor for the dashboard migrations API client */
  get api() {
    return api;
  }

  /** Returns any missing capabilities for the user to use this feature */
  getMissingCapabilities(level) {
    const getMissingCapabilities = (0, _capabilities.getMissingCapabilitiesChecker)(_capabilities2.requiredDashboardMigrationCapabilities);
    return getMissingCapabilities(this.core.application.capabilities, level);
  }

  /** Checks if the service is available based on the `license`, `capabilities` and `experimentalFeatures` */
  isAvailable() {
    const {
      automaticDashboardsMigration,
      siemMigrationsDisabled
    } = _experimental_features_service.ExperimentalFeaturesService.get();
    return automaticDashboardsMigration && !siemMigrationsDisabled && _use_license.licenseService.isEnterprise() && !this.hasMissingCapabilities('minimum');
  }

  /** Adds dashboards to a dashboard migration, batching the requests to avoid hitting the max payload size limit of the API */
  async addDashboardsToMigration(migrationId, dashboards) {
    const dashboardsCount = dashboards.length;
    if (dashboardsCount === 0) {
      throw new Error(i18n.EMPTY_DASHBOARDS_ERROR);
    }

    // Batching creation to avoid hitting the max payload size limit of the API
    const batches = [];
    for (let i = 0; i < dashboardsCount; i += CREATE_MIGRATION_BODY_BATCH_SIZE) {
      const dashboardsBatch = dashboards.slice(i, i + CREATE_MIGRATION_BODY_BATCH_SIZE);
      batches.push(api.addDashboardsToDashboardMigration({
        migrationId,
        body: dashboardsBatch
      }));
    }
    await Promise.all(batches);
  }

  /** Creates a dashboard migration with a name and adds the dashboards to it, returning the migration ID */
  async createDashboardMigration(data, migrationName) {
    const dashboardsCount = data.length;
    if (dashboardsCount === 0) {
      const emptyDashboardError = new Error(i18n.EMPTY_DASHBOARDS_ERROR);
      this.telemetry.reportSetupMigrationCreated({
        count: dashboardsCount,
        error: emptyDashboardError
      });
      throw emptyDashboardError;
    }
    try {
      // create the migration
      const {
        migration_id: migrationId
      } = await api.createDashboardMigration({
        name: migrationName
      });
      await this.addDashboardsToMigration(migrationId, data);
      this.telemetry.reportSetupMigrationCreated({
        migrationId,
        count: dashboardsCount
      });
      return migrationId;
    } catch (error) {
      this.telemetry.reportSetupMigrationCreated({
        count: dashboardsCount,
        error
      });
      throw error;
    }
  }

  /** Upserts resources for a dashboard migration, batching the requests to avoid hitting the max payload size limit of the API */
  async upsertMigrationResources(migrationId, body) {
    const count = body.length;
    if (count === 0) {
      throw new Error(i18n.EMPTY_DASHBOARDS_ERROR);
    }
    const type = body[0].type;

    // Batching creation to avoid hitting the max payload size limit of the API
    const batches = [];
    for (let i = 0; i < count; i += CREATE_MIGRATION_BODY_BATCH_SIZE) {
      const bodyBatch = body.slice(i, i + CREATE_MIGRATION_BODY_BATCH_SIZE);
      batches.push(api.upsertDashboardMigrationResources({
        migrationId,
        body: bodyBatch
      }));
    }
    await Promise.all(batches).then(() => {
      this.telemetry.reportSetupResourceUploaded({
        migrationId,
        type,
        count
      });
    }).catch(error => {
      this.telemetry.reportSetupResourceUploaded({
        migrationId,
        type,
        count,
        error
      });
      throw error;
    });
  }

  /** Starts a dashbaord migration task and waits for the task to start running */
  async startDashboardMigration(migrationId, retry, settings) {
    var _settings$connectorId;
    const missingCapabilities = this.getMissingCapabilities('all');
    if (missingCapabilities.length > 0) {
      this.core.notifications.toasts.add((0, _missing_capabilities_notification.getMissingCapabilitiesToast)(missingCapabilities, this.core));
      return {
        started: false
      };
    }
    const connectorId = (_settings$connectorId = settings === null || settings === void 0 ? void 0 : settings.connectorId) !== null && _settings$connectorId !== void 0 ? _settings$connectorId : this.connectorIdStorage.get();
    if (!connectorId) {
      this.core.notifications.toasts.add((0, _no_connector_notification.getNoConnectorToast)(this.core));
      return {
        started: false
      };
    }
    const params = {
      migrationId,
      settings: {
        connectorId
      },
      retry
    };
    const traceOptions = this.traceOptionsStorage.get();
    if (traceOptions) {
      params.langSmithOptions = {
        project_name: traceOptions.langSmithProject,
        api_key: traceOptions.langSmithApiKey
      };
    }
    try {
      const result = await api.startDashboardMigration(params);

      // Should take a few seconds to start the task, so we poll until it is running
      await this.migrationTaskPollingUntil(migrationId, ({
        status
      }) => status === _constants.SiemMigrationTaskStatus.RUNNING, {
        sleepSecs: _constants2.START_STOP_POLLING_SLEEP_SECONDS,
        timeoutSecs: 90
      } // wait up to 90 seconds for the task to start
      );
      this.startPolling();
      this.telemetry.reportStartTranslation(params);
      return result;
    } catch (error) {
      this.telemetry.reportStartTranslation({
        ...params,
        error
      });
      throw error;
    }
  }

  /** Stops a running dashboard migration task and waits for the task to completely stop */
  async stopDashboardMigration(migrationId) {
    const missingCapabilities = this.getMissingCapabilities('all');
    if (missingCapabilities.length > 0) {
      this.core.notifications.toasts.add((0, _missing_capabilities_notification.getMissingCapabilitiesToast)(missingCapabilities, this.core));
      return {
        stopped: false
      };
    }
    const params = {
      migrationId
    };
    try {
      const result = await api.stopDashboardMigration(params);

      // Should take a few seconds to stop the task, so we poll until it is not running anymore
      await this.migrationTaskPollingUntil(migrationId, ({
        status
      }) => status !== _constants.SiemMigrationTaskStatus.RUNNING,
      // may be STOPPED, FINISHED or INTERRUPTED
      {
        sleepSecs: _constants2.START_STOP_POLLING_SLEEP_SECONDS,
        timeoutSecs: 90
      } // wait up to 90 seconds for the task to stop
      );
      this.telemetry.reportStopTranslation(params);
      return result;
    } catch (error) {
      this.telemetry.reportStopTranslation({
        ...params,
        error
      });
      throw error;
    }
  }
  async startMigrationFromStats(connectorId, taskStats) {
    const params = {
      migrationId: taskStats.id,
      settings: {
        connectorId
      }
    };
    await api.startDashboardMigration(params).then(() => {
      this.telemetry.reportStartTranslation(params);
    }).catch(error => {
      this.telemetry.reportStartTranslation({
        ...params,
        error
      });
      throw error;
    });
  }
  async fetchMigrationStats({
    migrationId
  }) {
    const stats = await api.getDashboardMigrationStats({
      migrationId
    });
    return stats;
  }
  async fetchMigrationsStatsAll(params = {}) {
    const allStats = await api.getDashboardMigrationAllStats(params);
    return allStats;
  }
  sendFinishedMigrationNotification(taskStats) {
    (0, _success_notification.raiseSuccessToast)(taskStats, this.core);
  }

  /** Deletes a dashboard migration by its ID, refreshing the stats to remove it from the list */
  async deleteMigration(migrationId) {
    try {
      await api.deleteDashboardMigration({
        migrationId
      });

      // Refresh stats to remove the deleted migration from the list. All UI observables will be updated automatically
      await this.getMigrationsStats();
      this.telemetry.reportSetupMigrationDeleted({
        migrationId
      });
      return migrationId;
    } catch (error) {
      this.telemetry.reportSetupMigrationDeleted({
        migrationId,
        error
      });
      throw error;
    }
  }
}
exports.SiemDashboardMigrationsService = SiemDashboardMigrationsService;