"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SampleDataManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/task-manager-plugin/server");
var _inferenceCommon = require("@kbn/inference-common");
var _common = require("../../../common");
var _artifact_manager = require("../artifact_manager");
var _index_manager = require("../index_manager");
var _saved_objects_manager = require("../saved_objects_manager");
var _install_sample_data = require("../../tasks/install_sample_data");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SampleDataManager {
  constructor({
    artifactsFolder,
    logger,
    artifactRepositoryUrl,
    elserInferenceId,
    kibanaVersion,
    isServerlessPlatform,
    taskManager
  }) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "artifactManager", void 0);
    (0, _defineProperty2.default)(this, "indexManager", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsManager", void 0);
    (0, _defineProperty2.default)(this, "taskManager", void 0);
    (0, _defineProperty2.default)(this, "isInstalling", false);
    this.log = logger;
    this.taskManager = taskManager;
    this.artifactManager = new _artifact_manager.ArtifactManager({
      artifactsFolder,
      artifactRepositoryUrl,
      kibanaVersion,
      logger: this.log
    });
    this.indexManager = new _index_manager.IndexManager({
      elserInferenceId: elserInferenceId || _inferenceCommon.defaultInferenceEndpoints.ELSER,
      logger: this.log,
      isServerlessPlatform
    });
    this.savedObjectsManager = new _saved_objects_manager.SavedObjectsManager({
      logger: this.log
    });
  }
  async installSampleData({
    sampleType,
    esClient,
    soClient,
    soImporter,
    abortController
  }) {
    this.log.info(`Installing sample data for [${sampleType}]`);
    let archive;
    const indexName = (0, _common.getSampleDataIndexName)(sampleType);
    try {
      const status = await this.getSampleDataStatus({
        sampleType,
        esClient,
        soClient
      });
      if (status.status === 'installed' || this.isInstalling) {
        this.log.warn(`Sample data already installed for [${sampleType}]`);
        return {
          indexName,
          dashboardId: status.dashboardId
        };
      }
      this.isInstalling = true;
      const {
        archive: artifactsArchive,
        manifest,
        mappings
      } = await this.artifactManager.prepareArtifact(sampleType, abortController);
      archive = artifactsArchive;
      await this.indexManager.createAndPopulateIndex({
        indexName,
        mappings,
        manifest,
        archive,
        esClient,
        abortController
      });
      this.log.info(`Sample data installation successful for [${sampleType}]`);
      const {
        dashboardId
      } = await this.savedObjectsManager.importSavedObjects(soImporter, sampleType);
      return {
        indexName,
        dashboardId
      };
    } catch (error) {
      await this.indexManager.deleteIndex({
        indexName,
        esClient
      });
      await this.savedObjectsManager.deleteSavedObjects(soClient, sampleType);
      this.log.error(`Sample data installation failed for [${sampleType}]: ${(error === null || error === void 0 ? void 0 : error.message) || error}`);
      throw error;
    } finally {
      try {
        var _archive;
        (_archive = archive) === null || _archive === void 0 ? void 0 : _archive.close();
      } catch (e) {
        this.log.error(`Failed to close archive: ${(e === null || e === void 0 ? void 0 : e.message) || e}`);
      }
      await this.artifactManager.cleanup();
      this.isInstalling = false;
    }
  }
  async removeSampleData({
    sampleType,
    esClient,
    soClient
  }) {
    const indexName = (0, _common.getSampleDataIndexName)(sampleType);
    await this.indexManager.deleteIndex({
      indexName,
      esClient
    });
    if (soClient) {
      await this.savedObjectsManager.deleteSavedObjects(soClient, sampleType);
    }
  }
  async getSampleDataStatus({
    sampleType,
    esClient,
    soClient
  }) {
    try {
      if (this.taskManager) {
        const taskId = (0, _install_sample_data.getInstallTaskId)(sampleType);
        try {
          var _task$state;
          const task = await this.taskManager.get(taskId);
          const taskState = (_task$state = task.state) !== null && _task$state !== void 0 ? _task$state : {};
          const runAtTime = task.runAt ? new Date(task.runAt).getTime() : undefined;
          const retryAtTime = task.retryAt ? new Date(task.retryAt).getTime() : undefined;
          if (task.status === _server.TaskStatus.Failed || task.status === _server.TaskStatus.DeadLetter || taskState.status === 'error') {
            var _taskRunError, _error;
            const taskErrorMessage = taskState.errorMessage || ((_taskRunError = task.taskRunError) === null || _taskRunError === void 0 ? void 0 : _taskRunError.message) || ((_error = task.error) === null || _error === void 0 ? void 0 : _error.message);
            return {
              status: _common.InstallationStatus.Error,
              taskId,
              error: taskErrorMessage
            };
          }
          const isInFlight = task.status === _server.TaskStatus.Claiming || task.status === _server.TaskStatus.Running;
          const hasPendingRun = task.status === _server.TaskStatus.Idle && (runAtTime !== undefined || retryAtTime !== undefined);
          if ((isInFlight || hasPendingRun || taskState.status === 'pending') && taskState.status !== 'completed') {
            return {
              status: _common.InstallationStatus.Installing,
              taskId
            };
          }
        } catch (error) {
          var _statusCode, _output;
          const errorMessage = error instanceof Error ? error.message : String(error);
          const statusCode = (_statusCode = error.statusCode) !== null && _statusCode !== void 0 ? _statusCode : (_output = error.output) === null || _output === void 0 ? void 0 : _output.statusCode;
          if (statusCode && statusCode !== 404) {
            this.log.error(`Failed to check sample data task status for [${sampleType}]: ${errorMessage}`);
          }
          this.log.debug(`Task ${taskId} not found or error getting it: ${errorMessage}`);
        }
      }
      const isInstalled = await this.isSampleDataInstalled({
        sampleType,
        esClient
      });
      if (isInstalled) {
        const indexName = (0, _common.getSampleDataIndexName)(sampleType);
        const dashboardId = await this.savedObjectsManager.getDashboardId(soClient, sampleType);
        return {
          status: _common.InstallationStatus.Installed,
          indexName,
          dashboardId
        };
      }
      return {
        status: _common.InstallationStatus.Uninstalled
      };
    } catch (error) {
      this.log.warn(`Failed to check sample data status for [${sampleType}]: ${error.message}`);
      return {
        status: _common.InstallationStatus.Uninstalled
      };
    }
  }
  async isSampleDataInstalled({
    sampleType,
    esClient
  }) {
    const indexName = (0, _common.getSampleDataIndexName)(sampleType);
    try {
      return await this.indexManager.hasIndex({
        indexName,
        esClient
      });
    } catch (error) {
      this.log.warn(`Failed to check if sample data is installed for [${sampleType}]: ${error.message}`);
      return false;
    }
  }
}
exports.SampleDataManager = SampleDataManager;