"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkUpdateTagsRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _constants = require("../../common/constants");
var _constants2 = require("../alert_data_client/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bodySchema = _configSchema.schema.object({
  index: _configSchema.schema.string({
    maxLength: _constants2.MAX_INDEX_NAME
  }),
  add: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
    minSize: 1,
    maxSize: _constants2.MAX_TAGS_TO_UPDATE
  })),
  remove: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
    minSize: 1,
    maxSize: _constants2.MAX_TAGS_TO_UPDATE
  })),
  alertIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
    minSize: 1,
    maxSize: _constants2.MAX_ALERT_IDS_PER_REQUEST
  })),
  query: _configSchema.schema.maybe(_configSchema.schema.string({
    maxLength: _constants2.MAX_QUERY_LENGTH
  }))
});
const bulkUpdateTagsRoute = router => {
  router.post({
    path: `${_constants.BASE_RAC_ALERTS_API_PATH}/tags`,
    validate: {
      body: bodySchema
    },
    security: {
      authz: {
        requiredPrivileges: ['rac']
      }
    },
    options: {
      access: 'internal'
    }
  }, async (context, req, response) => {
    try {
      const racContext = await context.rac;
      const alertsClient = await racContext.getAlertsClient();
      const {
        query,
        alertIds,
        index,
        add,
        remove
      } = req.body;
      if (alertIds && query) {
        return response.badRequest({
          body: {
            message: 'Cannot specify both alertIds and query'
          }
        });
      }
      if (add == null && remove == null) {
        return response.badRequest({
          body: {
            message: 'No tags to add or remove were provided'
          }
        });
      }
      const res = await alertsClient.bulkUpdateTags({
        alertIds,
        add,
        remove,
        query,
        index
      });
      return response.multiStatus({
        body: res
      });
    } catch (exc) {
      const err = (0, _securitysolutionEsUtils.transformError)(exc);
      const contentType = {
        'content-type': 'application/json'
      };
      const defaultedHeaders = {
        ...contentType
      };
      return response.customError({
        headers: defaultedHeaders,
        statusCode: err.statusCode,
        body: {
          message: err.message,
          attributes: {
            success: false
          }
        }
      });
    }
  });
};
exports.bulkUpdateTagsRoute = bulkUpdateTagsRoute;