"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enrichMetricFields = enrichMetricFields;
exports.generateMapKey = generateMapKey;
exports.sampleMetricMetadata = sampleMetricMetadata;
var _otelSemanticConventions = require("@kbn/otel-semantic-conventions");
var _esQuery = require("@kbn/es-query");
var _extract_dimensions = require("../dimensions/extract_dimensions");
var _normalize_unit = require("./normalize_unit");
var _constants = require("../../../common/fields/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function isErrorResponseBase(subject) {
  return typeof subject === 'object' && subject !== null && 'error' in subject;
}
function generateMapKey(indexName, fieldName) {
  return `${fieldName}>${indexName}`;
}
function buildMetricMetadataMap(response, metricFields, logger) {
  const entries = new Map(metricFields.map(({
    name,
    index: indexName
  }, index) => {
    var _searchResult$hits$hi, _searchResult$hits$hi2, _searchResult$hits$hi3, _searchResult$hits$hi4;
    const searchResult = response.responses[index];
    const mapKey = generateMapKey(indexName, name);
    if (isErrorResponseBase(searchResult)) {
      logger.error(`Error sampling document for metric ${name}: ${searchResult.error}`);
      return [mapKey, {
        dimensions: [],
        totalHits: 0
      }];
    }
    if (!(searchResult !== null && searchResult !== void 0 && (_searchResult$hits$hi = searchResult.hits.hits) !== null && _searchResult$hits$hi !== void 0 && _searchResult$hits$hi.length)) {
      return [mapKey, {
        dimensions: [],
        totalHits: 0
      }];
    }
    const fields = (_searchResult$hits$hi2 = searchResult.hits.hits[0].fields) !== null && _searchResult$hits$hi2 !== void 0 ? _searchResult$hits$hi2 : {};
    const {
      dimensions,
      unitFromSample
    } = Object.entries(fields).reduce((acc, [fieldName, fieldValue]) => {
      if (fieldName === _otelSemanticConventions.semconvFlat.unit.name) {
        const value = Array.isArray(fieldValue) ? fieldValue[0] : fieldValue;
        if (typeof value === 'string') {
          acc.unitFromSample = value;
        }
        return acc;
      } else {
        acc.dimensions.push(fieldName);
      }
      return acc;
    }, {
      dimensions: [],
      unitFromSample: undefined
    });
    return [mapKey, {
      dimensions,
      unitFromSample,
      totalHits: (_searchResult$hits$hi3 = searchResult === null || searchResult === void 0 ? void 0 : (_searchResult$hits$hi4 = searchResult.hits.hits) === null || _searchResult$hits$hi4 === void 0 ? void 0 : _searchResult$hits$hi4.length) !== null && _searchResult$hits$hi3 !== void 0 ? _searchResult$hits$hi3 : 0
    }];
  }));
  return new Map(entries);
}
const buildFilters = (filterEntries, dimensions) => {
  const dimensionMap = new Map(dimensions.map(dimension => [dimension.name, dimension]));
  return filterEntries.flatMap(([dimensionName, values]) => {
    const dimension = dimensionMap.get(dimensionName);
    if (!dimension) {
      return (0, _esQuery.termsQuery)(dimensionName, values);
    }
    return (0, _esQuery.termsQuery)(dimensionName, _constants.NUMERIC_TYPES.includes(dimension.type) ? values.map(Number).filter(Boolean) : values);
  });
};
async function sampleMetricMetadata({
  esClient,
  metricFields,
  logger,
  timerange: {
    from,
    to
  },
  filters
}) {
  if (metricFields.length === 0) {
    return new Map();
  }
  const filterEntries = Object.entries(filters);
  try {
    const body = [];
    for (const {
      name: field,
      index,
      dimensions
    } of metricFields) {
      body.push({
        index
      });
      // Body for each search
      body.push({
        size: 1,
        terminate_after: 1,
        query: {
          bool: {
            filter: [{
              exists: {
                field
              }
            }, ...(0, _esQuery.dateRangeQuery)(from, to), ...(filterEntries.length > 0 ? [{
              bool: {
                should: buildFilters(filterEntries, dimensions),
                minimum_should_match: 1
              }
            }] : [])]
          }
        },
        _source: false,
        fields: dimensions.map(dimension => dimension.name).concat(_otelSemanticConventions.semconvFlat.unit.name)
      });
    }
    const response = await esClient.msearch('sample_metrics_documents', {
      body
    });
    return buildMetricMetadataMap(response, metricFields, logger);
  } catch (error) {
    const metricMetadataMap = new Map();
    for (const {
      name
    } of metricFields) {
      metricMetadataMap.set(name, {
        dimensions: [],
        totalHits: 0
      });
    }
    return metricMetadataMap;
  }
}
async function enrichMetricFields({
  esClient,
  metricFields,
  indexFieldCapsMap,
  logger,
  timerange,
  filters = {}
}) {
  if (metricFields.length === 0) {
    return metricFields;
  }
  const metricMetadataMap = await sampleMetricMetadata({
    esClient,
    metricFields,
    logger,
    timerange,
    filters
  });
  const uniqueDimensionSets = new Map();
  return metricFields.map(field => {
    var _uniqueDimensionSets$, _field$unit;
    const {
      dimensions,
      unitFromSample,
      totalHits
    } = metricMetadataMap.get(generateMapKey(field.index, field.name)) || {};
    const fieldCaps = indexFieldCapsMap.get(field.index);
    if ((!dimensions || dimensions.length === 0) && totalHits === 0) {
      return {
        ...field,
        dimensions: [],
        noData: true
      };
    }
    const cacheKey = dimensions ? [...(dimensions || [])].sort().join(',') : undefined;
    if (cacheKey && !uniqueDimensionSets.has(cacheKey) && fieldCaps) {
      uniqueDimensionSets.set(cacheKey, (0, _extract_dimensions.extractDimensions)(fieldCaps, dimensions));
    }
    return {
      ...field,
      dimensions: cacheKey ? (_uniqueDimensionSets$ = uniqueDimensionSets.get(cacheKey)) !== null && _uniqueDimensionSets$ !== void 0 ? _uniqueDimensionSets$ : [] : [],
      noData: false,
      unit: (0, _normalize_unit.normalizeUnit)({
        fieldName: field.name,
        unit: (_field$unit = field.unit) !== null && _field$unit !== void 0 ? _field$unit : unitFromSample
      })
    };
  });
}