"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.archiveMaintenanceWindowRoute = void 0;
var _lib = require("../../../../lib");
var _common = require("../../../../../../common");
var _archive = require("../../../../schemas/maintenance_window/external/request/archive");
var _response = require("../../../../schemas/maintenance_window/external/response");
var _transforms = require("../common/transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const archiveMaintenanceWindowRoute = (router, licenseState) => {
  router.post({
    path: _common.ARCHIVE_MAINTENANCE_WINDOW_API_PATH,
    validate: {
      request: {
        params: _archive.archiveMaintenanceWindowRequestParamsSchemaV1
      },
      response: {
        200: {
          body: () => _response.maintenanceWindowResponseSchemaV1,
          description: 'Indicates a successful call.'
        },
        400: {
          description: 'Indicates an invalid schema or parameters.'
        },
        403: {
          description: 'Indicates that this call is forbidden.'
        },
        404: {
          description: 'Indicates a maintenance window with the given ID does not exist.'
        }
      }
    },
    security: {
      authz: {
        requiredPrivileges: [`${_common.MAINTENANCE_WINDOW_API_PRIVILEGES.WRITE_MAINTENANCE_WINDOW}`]
      }
    },
    options: {
      access: 'public',
      summary: 'Archive a maintenance window.',
      tags: ['oas-tag:maintenance-window'],
      availability: {
        since: '9.1.0',
        stability: 'stable'
      }
    }
  }, router.handleLegacyErrors((0, _lib.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    licenseState.ensureLicenseForMaintenanceWindow();
    const params = req.params;
    const maintenanceWindowClient = (await context.maintenanceWindow).getMaintenanceWindowClient();
    const maintenanceWindow = await maintenanceWindowClient.archive({
      id: params.id,
      archive: true
    });
    const response = {
      body: (0, _transforms.transformInternalMaintenanceWindowToExternalV1)(maintenanceWindow)
    };
    return res.ok(response);
  })));
};
exports.archiveMaintenanceWindowRoute = archiveMaintenanceWindowRoute;