"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsUpdateAPIRoute = void 0;
var _boom = require("@hapi/boom");
var _utils = require("@kbn/lens-embeddable-utils/config_builder/utils");
var _constants = require("../../../../common/constants");
var _schema = require("./schema");
var _utils2 = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsUpdateAPIRoute = (router, {
  contentManagement,
  builder
}) => {
  const updateRoute = router.put({
    path: `${_constants.LENS_VIS_API_PATH}/{id}`,
    access: _constants.LENS_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Update Lens visualization',
    description: 'Update an existing Lens visualization.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  updateRoute.addVersion({
    version: _constants.LENS_API_VERSION,
    validate: {
      request: {
        params: _schema.lensUpdateRequestParamsSchema,
        body: _schema.lensUpdateRequestBodySchema,
        query: _schema.lensUpdateRequestQuerySchema
      },
      response: {
        200: {
          body: () => _schema.lensUpdateResponseBodySchema,
          description: 'Ok'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        404: {
          description: 'Resource not found'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    const requestBodyData = req.body;
    if ((0, _utils.isLensLegacyAttributes)(requestBodyData) && !requestBodyData.visualizationType) {
      throw new Error('visualizationType is required');
    }

    // TODO fix IContentClient to type this client based on the actual
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_constants.LENS_CONTENT_TYPE);

    // Note: these types are to enforce loose param typings of client methods
    const {
      references,
      ...data
    } = (0, _utils2.getLensRequestConfig)(builder, req.body);
    const options = {
      ...req.query,
      references
    };
    try {
      const {
        result
      } = await client.update(req.params.id, data, options);
      if (result.item.error) {
        throw result.item.error;
      }
      const responseItem = (0, _utils2.getLensResponseItem)(builder, result.item);
      return res.ok({
        body: responseItem
      });
    } catch (error) {
      if ((0, _boom.isBoom)(error)) {
        if (error.output.statusCode === 404) {
          return res.notFound({
            body: {
              message: `A Lens visualization with id [${req.params.id}] was not found.`
            }
          });
        }
        if (error.output.statusCode === 403) {
          return res.forbidden();
        }
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
  });
};
exports.registerLensVisualizationsUpdateAPIRoute = registerLensVisualizationsUpdateAPIRoute;