"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDropProps = getDropProps;
exports.getField = getField;
exports.getNewOperation = getNewOperation;
var _types_guards = require("../../../../types_guards");
var _operations = require("../../operations");
var _utils = require("../../../../utils");
var _pure_utils = require("../../pure_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const operationDisplay = (0, _operations.getOperationDisplay)();
function getNewOperation(field, filterOperations, targetColumn, prioritizedOperation, alreadyUsedOperations) {
  if (!field) {
    return;
  }
  const newOperations = (0, _operations.getOperationTypesForField)(field, filterOperations, alreadyUsedOperations);
  if (!newOperations.length) {
    return;
  }
  // Detects if we can change the field only, otherwise change field + operation
  const shouldOperationPersist = targetColumn && newOperations.includes(targetColumn.operationType);
  if (shouldOperationPersist) {
    return targetColumn.operationType;
  }
  const existsPrioritizedOperation = prioritizedOperation && newOperations.includes(prioritizedOperation);
  return existsPrioritizedOperation ? prioritizedOperation : newOperations[0];
}
function getField(column, dataView) {
  if (!column) {
    return;
  }
  const field = (0, _pure_utils.hasField)(column) && dataView.getFieldByName(column.sourceField) || undefined;
  return field;
}
function getDropProps(props) {
  const {
    state,
    source,
    target,
    indexPatterns
  } = props;
  if (!source) {
    return;
  }
  const targetProps = {
    ...target,
    column: state.layers[target.layerId].columns[target.columnId],
    dataView: indexPatterns[state.layers[target.layerId].indexPatternId]
  };
  if ((0, _utils.isDraggedDataViewField)(source)) {
    return getDropPropsForField({
      ...props,
      source,
      target: targetProps
    });
  }
  if ((0, _types_guards.isOperation)(source)) {
    var _state$layers$source$, _state$layers$source$2;
    const sourceProps = {
      ...source,
      column: (_state$layers$source$ = state.layers[source.layerId]) === null || _state$layers$source$ === void 0 ? void 0 : _state$layers$source$.columns[source.columnId],
      dataView: indexPatterns[(_state$layers$source$2 = state.layers[source.layerId]) === null || _state$layers$source$2 === void 0 ? void 0 : _state$layers$source$2.indexPatternId]
    };
    if (!sourceProps.column) {
      return;
    }
    if (target.columnId !== source.columnId && targetProps.dataView === sourceProps.dataView) {
      var _targetProps$filterOp;
      if ((0, _utils.isOperationFromTheSameGroup)(source, target)) {
        return !targetProps.column ? {
          dropTypes: ['duplicate_compatible']
        } : {
          dropTypes: ['reorder']
        };
      }
      if ((_targetProps$filterOp = targetProps.filterOperations) !== null && _targetProps$filterOp !== void 0 && _targetProps$filterOp.call(targetProps, sourceProps === null || sourceProps === void 0 ? void 0 : sourceProps.column)) {
        return getDropPropsForCompatibleGroup(sourceProps, targetProps);
      } else if (hasTheSameField(sourceProps.column, targetProps.column)) {
        return;
      } else {
        return getDropPropsFromIncompatibleGroup(sourceProps, targetProps);
      }
    }
  }
}
function hasTheSameField(sourceColumn, targetColumn) {
  const targetFields = targetColumn ? (0, _operations.getCurrentFieldsForOperation)(targetColumn) : [];
  const sourceFields = new Set((0, _operations.getCurrentFieldsForOperation)(sourceColumn));
  return targetFields.length === sourceFields.size && targetFields.every(field => sourceFields.has(field));
}
function getDropPropsForField({
  state,
  source,
  target,
  indexPatterns
}) {
  const targetColumn = state.layers[target.layerId].columns[target.columnId];
  const isTheSameIndexPattern = state.layers[target.layerId].indexPatternId === source.indexPatternId;
  const newOperation = getNewOperation(source.field, target.filterOperations, targetColumn);
  if (isTheSameIndexPattern && newOperation) {
    const nextLabel = operationDisplay[newOperation].displayName;
    if (!targetColumn) {
      return {
        dropTypes: ['field_add'],
        nextLabel
      };
    } else if ((0, _pure_utils.hasField)(targetColumn) && targetColumn.sourceField !== source.field.name || !(0, _pure_utils.hasField)(targetColumn)) {
      const layerDataView = indexPatterns[state.layers[target.layerId].indexPatternId];
      return (0, _pure_utils.hasField)(targetColumn) && layerDataView && (0, _operations.hasOperationSupportForMultipleFields)(layerDataView, targetColumn, undefined, source.field) ? {
        dropTypes: ['field_replace', 'field_combine']
      } : {
        dropTypes: ['field_replace'],
        nextLabel
      };
    }
  }
  return;
}
function getDropPropsForCompatibleGroup(sourceProps, targetProps) {
  var _sourceProps$filterOp;
  if (!targetProps.column) {
    return {
      dropTypes: ['move_compatible', 'duplicate_compatible']
    };
  }
  const canSwap = (_sourceProps$filterOp = sourceProps.filterOperations) === null || _sourceProps$filterOp === void 0 ? void 0 : _sourceProps$filterOp.call(sourceProps, targetProps.column);
  const swapType = canSwap ? ['swap_compatible'] : [];
  const dropTypes = ['replace_compatible', 'replace_duplicate_compatible', ...swapType];
  if (!targetProps.dataView || !(0, _pure_utils.hasField)(targetProps.column)) {
    return {
      dropTypes
    };
  }
  if ((0, _operations.hasOperationSupportForMultipleFields)(targetProps.dataView, targetProps.column, sourceProps.column)) {
    dropTypes.push('combine_compatible');
  }
  return {
    dropTypes
  };
}
function getDropPropsFromIncompatibleGroup(sourceProps, targetProps) {
  if (!targetProps.dataView || !sourceProps.column) {
    return;
  }
  const sourceField = getField(sourceProps.column, sourceProps.dataView);
  const newOperationForSource = getNewOperation(sourceField, targetProps.filterOperations, targetProps.column);
  if (newOperationForSource) {
    const targetField = getField(targetProps.column, targetProps.dataView);
    const canSwap = Boolean(getNewOperation(targetField, sourceProps.filterOperations, sourceProps.column));
    const dropTypes = [];
    if (targetProps.column) {
      dropTypes.push('replace_incompatible', 'replace_duplicate_incompatible');
      if (canSwap) {
        dropTypes.push('swap_incompatible');
      }
      if ((0, _operations.hasOperationSupportForMultipleFields)(targetProps.dataView, targetProps.column, sourceProps.column)) {
        dropTypes.push('combine_incompatible');
      }
    } else {
      dropTypes.push('move_incompatible', 'duplicate_incompatible');
    }
    return {
      dropTypes,
      nextLabel: operationDisplay[newOperationForSource].displayName
    };
  }
}