"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.xyStateSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _shared = require("../shared");
var _dataset = require("../dataset");
var _shared2 = require("./shared");
var _metric_ops = require("../metric_ops");
var _color = require("../color");
var _filter = require("../filter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Statistical functions that can be displayed in chart legend for data series
 */
const statisticsSchema = _configSchema.schema.oneOf([_configSchema.schema.oneOf([_configSchema.schema.literal('min'), _configSchema.schema.literal('max'), _configSchema.schema.literal('avg'), _configSchema.schema.literal('median'), _configSchema.schema.literal('range'), _configSchema.schema.literal('last_value'), _configSchema.schema.literal('last_non_null_value'), _configSchema.schema.literal('first_value'), _configSchema.schema.literal('first_non_null_value')]), _configSchema.schema.oneOf([_configSchema.schema.literal('difference'), _configSchema.schema.literal('difference_percentage'), _configSchema.schema.literal('count'), _configSchema.schema.literal('total'), _configSchema.schema.literal('standard_deviation'), _configSchema.schema.literal('variance'), _configSchema.schema.literal('distinct_count'), _configSchema.schema.literal('current_and_last_value')])], {
  meta: {
    description: 'Statistical functions that can be displayed in chart legend for data series'
  }
});

/**
 * Y-axis extent configuration defining how the axis bounds are calculated
 */
const yExtendSchema = _configSchema.schema.oneOf([_configSchema.schema.object({
  type: _configSchema.schema.literal('full'),
  integer_rounding: _configSchema.schema.maybe(_configSchema.schema.boolean())
}, {
  meta: {
    description: 'Full extent mode with optional integer rounding'
  }
}), _configSchema.schema.object({
  type: _configSchema.schema.literal('focus')
}, {
  meta: {
    description: 'Focus mode for tighter data bounds'
  }
}), _configSchema.schema.object({
  type: _configSchema.schema.literal('custom'),
  start: _configSchema.schema.number({
    meta: {
      description: 'Custom axis start value'
    }
  }),
  end: _configSchema.schema.number({
    meta: {
      description: 'Custom axis end value'
    }
  }),
  integer_rounding: _configSchema.schema.maybe(_configSchema.schema.boolean())
}, {
  meta: {
    description: 'Custom extent with explicit bounds'
  }
})], {
  meta: {
    description: 'Y-axis extent configuration defining how the axis bounds are calculated'
  }
});

/**
 * Y-axis scale type for data transformation
 */
const yScaleSchema = _configSchema.schema.oneOf([_configSchema.schema.literal('time'), _configSchema.schema.literal('linear'), _configSchema.schema.literal('log'), _configSchema.schema.literal('sqrt')], {
  meta: {
    description: 'Y-axis scale type for data transformation'
  }
});

/**
 * Common axis configuration properties shared across X and Y axes
 */
const sharedAxisSchema = {
  title: _configSchema.schema.maybe(_configSchema.schema.object({
    value: _configSchema.schema.string({
      meta: {
        description: 'Axis title text'
      }
    }),
    visible: _configSchema.schema.maybe(_configSchema.schema.boolean())
  }, {
    meta: {
      description: 'Axis title configuration'
    }
  })),
  ticks: _configSchema.schema.maybe(_configSchema.schema.boolean({
    meta: {
      description: 'Whether to show tick marks on the axis'
    }
  })),
  grid: _configSchema.schema.maybe(_configSchema.schema.boolean({
    meta: {
      description: 'Whether to show grid lines for this axis'
    }
  })),
  label_orientation: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('horizontal'), _configSchema.schema.literal('vertical'), _configSchema.schema.literal('angled')], {
    meta: {
      description: 'Orientation of the axis labels'
    }
  }))
};

/**
 * Complete Y-axis configuration including scale and extent
 */
const yAxisSchema = _configSchema.schema.object({
  ...sharedAxisSchema,
  scale: _configSchema.schema.maybe(yScaleSchema),
  extent: _configSchema.schema.maybe(yExtendSchema)
}, {
  meta: {
    description: 'Y-axis configuration with scale and bounds'
  }
});

/**
 * Chart types available for data layers in XY visualizations
 */
const xyDataLayerSharedSchema = {
  type: _configSchema.schema.oneOf([_configSchema.schema.literal('area'), _configSchema.schema.literal('area_percentage'), _configSchema.schema.literal('area_stacked'), _configSchema.schema.literal('bar'), _configSchema.schema.literal('bar_horizontal'), _configSchema.schema.literal('bar_horizontal_stacked'), _configSchema.schema.literal('bar_horizontal_percentage'), _configSchema.schema.literal('bar_percentage'), _configSchema.schema.literal('bar_stacked'), _configSchema.schema.literal('line')], {
    meta: {
      description: 'Chart type for the data layer'
    }
  })
};

/**
 * Common legend configuration properties for positioning and statistics
 */
const sharedLegendSchema = {
  visible: _configSchema.schema.maybe(_configSchema.schema.boolean({
    meta: {
      description: 'Whether to show the legend'
    }
  })),
  statistics: _configSchema.schema.maybe(_configSchema.schema.arrayOf(statisticsSchema, {
    meta: {
      description: 'Statistics to display in legend'
    }
  })),
  truncate_after_lines: _configSchema.schema.maybe(_configSchema.schema.number({
    min: 1,
    max: 5,
    meta: {
      description: 'Maximum lines before truncation'
    }
  }))
};

/**
 * Shared settings that apply to the entire XY chart visualization
 */
const xySharedSettings = {
  legend: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.object({
    ...sharedLegendSchema,
    inside: _configSchema.schema.maybe(_configSchema.schema.literal(false)),
    position: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('top'), _configSchema.schema.literal('bottom'), _configSchema.schema.literal('left'), _configSchema.schema.literal('right')])),
    size: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('small'), _configSchema.schema.literal('medium'), _configSchema.schema.literal('large'), _configSchema.schema.literal('xlarge')]))
  }, {
    meta: {
      description: 'External legend positioned outside the chart'
    }
  }), _configSchema.schema.object({
    ...sharedLegendSchema,
    inside: _configSchema.schema.literal(true),
    columns: _configSchema.schema.maybe(_configSchema.schema.number({
      min: 1,
      max: 5,
      meta: {
        description: 'Number of legend columns'
      }
    })),
    alignment: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('top_right'), _configSchema.schema.literal('bottom_right'), _configSchema.schema.literal('top_left'), _configSchema.schema.literal('bottom_left')], {
      meta: {
        description: 'Legend alignment inside the chart'
      }
    }))
  }, {
    meta: {
      description: 'Internal legend positioned inside the chart'
    }
  })])),
  fitting: _configSchema.schema.maybe(_configSchema.schema.object({
    type: _configSchema.schema.oneOf([_configSchema.schema.literal('none'), _configSchema.schema.literal('zero'), _configSchema.schema.literal('linear'), _configSchema.schema.literal('carry'), _configSchema.schema.literal('lookahead'), _configSchema.schema.literal('average'), _configSchema.schema.literal('nearest')], {
      meta: {
        description: 'Fitting function type for missing data'
      }
    }),
    dotted: _configSchema.schema.maybe(_configSchema.schema.boolean({
      meta: {
        description: 'Show fitted values as dotted lines'
      }
    })),
    end_value: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('none'), _configSchema.schema.literal('zero'), _configSchema.schema.literal('nearest')]))
  }, {
    meta: {
      description: 'Missing data interpolation configuration (only valid fitting types applied per chart type)'
    }
  })),
  axis: _configSchema.schema.maybe(_configSchema.schema.object({
    x: _configSchema.schema.maybe(_configSchema.schema.object({
      ...sharedAxisSchema,
      extent: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.object({
        type: _configSchema.schema.literal('full'),
        integer_rounding: _configSchema.schema.maybe(_configSchema.schema.boolean())
      }, {
        meta: {
          description: 'Full extent mode for X-axis'
        }
      }), _configSchema.schema.object({
        type: _configSchema.schema.literal('custom'),
        start: _configSchema.schema.number({
          meta: {
            description: 'Custom X-axis start value'
          }
        }),
        end: _configSchema.schema.number({
          meta: {
            description: 'Custom X-axis end value'
          }
        }),
        integer_rounding: _configSchema.schema.maybe(_configSchema.schema.boolean())
      }, {
        meta: {
          description: 'Custom X-axis extent'
        }
      })]))
    }, {
      meta: {
        description: 'X-axis (horizontal) configuration'
      }
    })),
    left: _configSchema.schema.maybe(yAxisSchema),
    right: _configSchema.schema.maybe(yAxisSchema)
  }, {
    meta: {
      description: 'Axis configuration for X, left Y, and right Y axes'
    }
  })),
  decorations: _configSchema.schema.maybe(_configSchema.schema.object({
    end_zones: _configSchema.schema.maybe(_configSchema.schema.boolean({
      meta: {
        description: 'Show end zones for partial buckets'
      }
    })),
    current_time_marker: _configSchema.schema.maybe(_configSchema.schema.boolean({
      meta: {
        description: 'Show current time marker line'
      }
    })),
    point_visibility: _configSchema.schema.maybe(_configSchema.schema.boolean({
      meta: {
        description: 'Show data points on lines'
      }
    })),
    line_interpolation: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('linear'), _configSchema.schema.literal('smooth'), _configSchema.schema.literal('stepped')])),
    minimum_bar_height: _configSchema.schema.maybe(_configSchema.schema.number({
      min: 0,
      meta: {
        description: 'Minimum bar height in pixels'
      }
    })),
    show_value_labels: _configSchema.schema.maybe(_configSchema.schema.boolean({
      meta: {
        description: 'Display value labels on data points'
      }
    })),
    fill_opacity: _configSchema.schema.maybe(_configSchema.schema.number({
      min: 0,
      max: 2,
      meta: {
        description: 'Area chart fill opacity (0-1 typical, max 2 for legacy)'
      }
    })),
    value_labels: _configSchema.schema.maybe(_configSchema.schema.boolean({
      meta: {
        description: 'Show value labels (alternative property)'
      }
    }))
  }, {
    meta: {
      description: 'Visual enhancements and styling options for the chart'
    }
  }))
};

/**
 * Data layer configuration for standard (non-ES|QL) queries with breakdown and metrics
 */
const xyDataLayerSchemaNoESQL = _configSchema.schema.object({
  ..._shared.layerSettingsSchema,
  ..._dataset.datasetSchema,
  ...xyDataLayerSharedSchema,
  breakdown_by: _configSchema.schema.maybe((0, _shared2.mergeAllBucketsWithChartDimensionSchema)(_configSchema.schema.object({
    collapse_by: _configSchema.schema.maybe(_shared.collapseBySchema),
    color: _configSchema.schema.maybe(_color.colorMappingSchema),
    aggregate_first: _configSchema.schema.maybe(_configSchema.schema.boolean({
      meta: {
        description: 'Whether to aggregate before splitting series'
      }
    }))
  }, {
    meta: {
      description: 'Split series configuration with color mapping'
    }
  }))),
  y: _configSchema.schema.arrayOf((0, _shared2.mergeAllMetricsWithChartDimensionSchemaWithRefBasedOps)(_configSchema.schema.object({
    axis: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('left'), _configSchema.schema.literal('right')])),
    color: _configSchema.schema.maybe(_color.staticColorSchema)
  }, {
    meta: {
      description: 'Y-axis metric configuration with axis assignment'
    }
  })), {
    meta: {
      description: 'Array of metrics to display on Y-axis'
    }
  }),
  x: _configSchema.schema.maybe((0, _shared2.mergeAllBucketsWithChartDimensionSchema)(_configSchema.schema.object({}, {
    meta: {
      description: 'X-axis bucket configuration'
    }
  })))
}, {
  meta: {
    description: 'Data layer for standard queries with metrics and buckets'
  }
});

/**
 * Data layer configuration for ES|QL queries with column-based metrics
 */
const xyDataLayerSchemaESQL = _configSchema.schema.object({
  ..._shared.layerSettingsSchema,
  ..._dataset.datasetEsqlTableSchema,
  ...xyDataLayerSharedSchema,
  breakdown_by: _configSchema.schema.maybe(_metric_ops.esqlColumnSchema),
  y: _configSchema.schema.arrayOf(_metric_ops.esqlColumnSchema.extends({
    axis: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('left'), _configSchema.schema.literal('right')])),
    color: _configSchema.schema.maybe(_color.staticColorSchema)
  }, {
    meta: {
      description: 'ES|QL column for Y-axis metric'
    }
  }), {
    meta: {
      description: 'Array of ES|QL columns for Y-axis metrics'
    }
  }),
  x: _configSchema.schema.maybe(_metric_ops.esqlColumnSchema)
}, {
  meta: {
    description: 'Data layer for ES|QL queries with column references'
  }
});
const getListOfAvailableIcons = description => _configSchema.schema.oneOf([_configSchema.schema.oneOf([_configSchema.schema.literal('empty'), _configSchema.schema.literal('asterisk'), _configSchema.schema.literal('alert'), _configSchema.schema.literal('bell'), _configSchema.schema.literal('bolt'), _configSchema.schema.literal('bug'), _configSchema.schema.literal('circle'), _configSchema.schema.literal('editorComment'), _configSchema.schema.literal('flag'), _configSchema.schema.literal('heart')]), _configSchema.schema.oneOf([_configSchema.schema.literal('mapMarker'), _configSchema.schema.literal('pinFilled'), _configSchema.schema.literal('starEmpty'), _configSchema.schema.literal('starFilled'), _configSchema.schema.literal('tag'), _configSchema.schema.literal('triangle')])]);
const STROKE_WIDTH_MIN = 1;
const STROKE_WIDTH_MAX = 10;

/**
 * Common visual configuration for reference lines
 */
const referenceLineLayerShared = {
  fill: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('above'), _configSchema.schema.literal('below')], {
    meta: {
      description: 'Fill direction for reference line'
    }
  })),
  text: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('none'), _configSchema.schema.literal('label')], {
    meta: {
      description: 'Text display option for the reference line'
    }
  })),
  icon: _configSchema.schema.maybe(getListOfAvailableIcons('Icon to display on the reference line')),
  stroke_width: _configSchema.schema.maybe(_configSchema.schema.number({
    meta: {
      description: 'Line width in pixels'
    },
    min: STROKE_WIDTH_MIN,
    max: STROKE_WIDTH_MAX
  })),
  stroke_dash: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('solid'), _configSchema.schema.literal('dashed'), _configSchema.schema.literal('dotted')], {
    meta: {
      description: 'Line style'
    }
  })),
  color: _configSchema.schema.maybe(_color.staticColorSchema),
  axis: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('bottom'), _configSchema.schema.literal('left'), _configSchema.schema.literal('right')], {
    defaultValue: 'left',
    meta: {
      description: 'Which axis the reference line applies to'
    }
  }))
};

/**
 * Reference line layer for standard queries with threshold values
 */
const referenceLineLayerSchemaNoESQL = _configSchema.schema.object({
  ..._shared.layerSettingsSchema,
  ..._dataset.datasetSchema,
  type: _configSchema.schema.literal('referenceLines'),
  thresholds: _configSchema.schema.arrayOf((0, _shared2.mergeAllMetricsWithChartDimensionSchemaWithStaticOps)(_configSchema.schema.object(referenceLineLayerShared, {
    meta: {
      description: 'Reference line threshold configuration'
    }
  })), {
    meta: {
      description: 'Array of reference line thresholds'
    },
    minSize: 1
  })
}, {
  meta: {
    description: 'Reference line layer for standard queries'
  }
});

/**
 * Reference line layer for ES|QL queries with column-based thresholds
 */
const referenceLineLayerSchemaESQL = _configSchema.schema.object({
  ..._shared.layerSettingsSchema,
  ..._dataset.datasetEsqlTableSchema,
  type: _configSchema.schema.literal('referenceLines'),
  thresholds: _configSchema.schema.arrayOf(_configSchema.schema.allOf([_metric_ops.esqlColumnSchema, _configSchema.schema.object(referenceLineLayerShared, {
    meta: {
      description: 'ES|QL reference line threshold'
    }
  })]), {
    meta: {
      description: 'Array of ES|QL-based reference line thresholds'
    },
    minSize: 1
  })
}, {
  meta: {
    description: 'Reference line layer for ES|QL queries'
  }
});

/**
 * Common properties for all annotation types
 */
const annotationEventShared = {
  color: _configSchema.schema.maybe(_color.staticColorSchema),
  hidden: _configSchema.schema.maybe(_configSchema.schema.boolean({
    meta: {
      description: 'Whether to hide the annotation'
    }
  }))
};

/**
 * Visual properties specific to point annotations
 */
const annotationPointShared = {
  ...annotationEventShared,
  icon: _configSchema.schema.maybe(getListOfAvailableIcons('Icon to display at the annotation point')),
  line: _configSchema.schema.maybe(_configSchema.schema.object({
    stroke_width: _configSchema.schema.number({
      meta: {
        description: 'Vertical line width in pixels'
      },
      min: STROKE_WIDTH_MIN,
      max: STROKE_WIDTH_MAX
    }),
    stroke_dash: _configSchema.schema.oneOf([_configSchema.schema.literal('solid'), _configSchema.schema.literal('dashed'), _configSchema.schema.literal('dotted')], {
      meta: {
        description: 'Vertical line style'
      }
    })
  }, {
    meta: {
      description: 'Vertical line configuration for point annotation'
    }
  }))
};

/**
 * Timestamp format for annotations (Unix timestamp or ISO date string)
 */
const annotationTimestampSchema = _configSchema.schema.oneOf([_configSchema.schema.number({
  meta: {
    description: 'Unix timestamp in milliseconds'
  }
}), _configSchema.schema.string({
  meta: {
    description: 'ISO date string'
  }
})]);

/**
 * Query-based annotation that finds events matching a filter
 */
const annotationQuery = _configSchema.schema.object({
  ...annotationPointShared,
  type: _configSchema.schema.literal('query'),
  query: _filter.filterSchema,
  time_field: _configSchema.schema.string({
    meta: {
      description: 'Field containing the timestamp'
    }
  }),
  label: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Label text for the annotation'
    }
  })),
  text: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('none'), _configSchema.schema.literal('label'), _configSchema.schema.object({
    type: _configSchema.schema.literal('field'),
    field: _configSchema.schema.string({
      meta: {
        description: 'Field name containing label text'
      }
    })
  }, {
    meta: {
      description: 'Text from document field'
    }
  })], {
    meta: {
      description: 'Text display option for the annotation'
    }
  })),
  extra_fields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string({
    meta: {
      description: 'Additional field to include in tooltip'
    }
  }), {
    meta: {
      description: 'Additional fields for annotation tooltip'
    }
  }))
}, {
  meta: {
    description: 'Annotation from query results matching a filter'
  }
});

/**
 * Manually specified point annotation at a specific timestamp
 */
const annotationManualEvent = _configSchema.schema.object({
  ...annotationPointShared,
  type: _configSchema.schema.literal('point'),
  timestamp: annotationTimestampSchema,
  label: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Label text for the annotation'
    }
  })),
  text: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('none'), _configSchema.schema.literal('label')], {
    meta: {
      description: 'Text display option for the annotation'
    }
  }))
}, {
  meta: {
    description: 'Manual point annotation at specific timestamp'
  }
});

/**
 * Manually specified range annotation spanning a time interval
 */
const annotationManualRange = _configSchema.schema.object({
  ...annotationEventShared,
  type: _configSchema.schema.literal('range'),
  interval: _configSchema.schema.object({
    from: annotationTimestampSchema,
    to: annotationTimestampSchema
  }, {
    meta: {
      description: 'Time range for annotation'
    }
  }),
  label: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Label text for the annotation'
    }
  })),
  fill: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('inside'), _configSchema.schema.literal('outside')], {
    defaultValue: 'inside',
    meta: {
      description: 'Fill direction for range'
    }
  }))
}, {
  meta: {
    description: 'Manual range annotation spanning time interval'
  }
});

/**
 * Annotation layer containing query-based, point, and range annotations
 */
const annotationLayerSchema = _configSchema.schema.object({
  ..._shared.ignoringGlobalFiltersSchemaRaw,
  ..._dataset.datasetSchema,
  type: _configSchema.schema.literal('annotations'),
  events: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([annotationQuery, annotationManualEvent, annotationManualRange]), {
    meta: {
      description: 'Array of annotation configurations'
    },
    minSize: 1
  })
}, {
  meta: {
    description: 'Layer containing annotations (query-based, points, and ranges)'
  }
});

/**
 * Complete XY chart state configuration with layers and visualization settings
 */
const xyStateSchema = exports.xyStateSchema = _configSchema.schema.object({
  type: _configSchema.schema.literal('xy'),
  ..._shared.sharedPanelInfoSchema,
  ...xySharedSettings,
  ..._shared.dslOnlyPanelInfoSchema,
  layers: _configSchema.schema.arrayOf(
  /**
   * Any valid XY chart layer type (data, reference line, or annotation)
   */
  _configSchema.schema.oneOf([xyDataLayerSchemaNoESQL, xyDataLayerSchemaESQL, referenceLineLayerSchemaNoESQL, referenceLineLayerSchemaESQL, annotationLayerSchema]), {
    minSize: 1,
    meta: {
      description: 'Chart layers (minimum 1 required)'
    }
  })
}, {
  meta: {
    description: 'Complete XY chart configuration'
  }
});