"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildMetric = buildMetric;
var _types = require("../types");
var _utils = require("../utils");
var _columns = require("../columns");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const ACCESSOR = 'metric_formula_accessor';
const HISTOGRAM_COLUMN_NAME = 'x_date_histogram';
const TRENDLINE_LAYER_ID = 'layer_0_trendline';
function getAccessorName(type) {
  return `${ACCESSOR}_${type}`;
}
function buildVisualizationState(config) {
  const layer = config;
  return {
    layerId: _types.DEFAULT_LAYER_ID,
    layerType: 'data',
    metricAccessor: ACCESSOR,
    color: layer.seriesColor,
    subtitle: layer.subtitle,
    showBar: false,
    ...(layer.querySecondaryMetric ? {
      secondaryMetricAccessor: getAccessorName('secondary')
    } : {}),
    ...(layer.queryMaxValue ? {
      maxAccessor: getAccessorName('max'),
      showBar: true
    } : {}),
    ...(layer.breakdown ? {
      breakdownByAccessor: getAccessorName('breakdown')
    } : {}),
    ...(layer.trendLine ? {
      trendlineLayerId: `${_types.DEFAULT_LAYER_ID}_trendline`,
      trendlineLayerType: 'metricTrendline',
      trendlineMetricAccessor: `${ACCESSOR}_trendline`,
      trendlineTimeAccessor: HISTOGRAM_COLUMN_NAME,
      ...(layer.querySecondaryMetric ? {
        trendlineSecondaryMetricAccessor: `${ACCESSOR}_secondary_trendline`
      } : {}),
      ...(layer.queryMaxValue ? {
        trendlineMaxAccessor: `${ACCESSOR}_max_trendline`
      } : {}),
      ...(layer.breakdown ? {
        trendlineBreakdownByAccessor: `${ACCESSOR}_breakdown_trendline`
      } : {})
    } : {})
  };
}
function buildFormulaLayer(layer, i, dataView) {
  const baseLayer = {
    columnOrder: [ACCESSOR, HISTOGRAM_COLUMN_NAME],
    columns: {
      [HISTOGRAM_COLUMN_NAME]: (0, _columns.getHistogramColumn)({
        options: {
          sourceField: dataView.timeFieldName,
          params: {
            interval: 'auto',
            includeEmptyRows: true
          }
        }
      })
    },
    sampling: 1
  };
  const layers = {
    [_types.DEFAULT_LAYER_ID]: {
      ...(0, _columns.getFormulaColumn)(ACCESSOR, (0, _utils.mapToFormula)(layer), dataView)
    },
    ...(layer.trendLine ? {
      [TRENDLINE_LAYER_ID]: {
        linkToLayers: [_types.DEFAULT_LAYER_ID],
        ...(0, _columns.getFormulaColumn)(`${ACCESSOR}_trendline`, (0, _utils.mapToFormula)(layer), dataView, baseLayer)
      }
    } : {})
  };
  const defaultLayer = layers[_types.DEFAULT_LAYER_ID];
  const trendLineLayer = layers[TRENDLINE_LAYER_ID];
  if (layer.breakdown) {
    const columnName = getAccessorName('breakdown');
    const breakdownColumn = (0, _columns.getBreakdownColumn)({
      options: layer.breakdown,
      dataView
    });
    (0, _utils.addLayerColumn)(defaultLayer, columnName, breakdownColumn, true);
    if (trendLineLayer) {
      (0, _utils.addLayerColumn)(trendLineLayer, `${columnName}_trendline`, breakdownColumn, true);
    }
  }
  if (layer.querySecondaryMetric) {
    const columnName = getAccessorName('secondary');
    const formulaColumn = (0, _columns.getFormulaColumn)(columnName, {
      formula: layer.querySecondaryMetric
    }, dataView);
    (0, _utils.addLayerFormulaColumns)(defaultLayer, formulaColumn);
    if (trendLineLayer) {
      (0, _utils.addLayerFormulaColumns)(trendLineLayer, formulaColumn, 'X0');
    }
  }
  if (layer.queryMaxValue) {
    const columnName = getAccessorName('max');
    const formulaColumn = (0, _columns.getFormulaColumn)(columnName, {
      formula: layer.queryMaxValue
    }, dataView);
    (0, _utils.addLayerFormulaColumns)(defaultLayer, formulaColumn);
    if (trendLineLayer) {
      (0, _utils.addLayerFormulaColumns)(trendLineLayer, formulaColumn, 'X0');
    }
  }
  return layers;
}
function getValueColumns(layer) {
  if (layer.breakdown && typeof layer.breakdown !== 'string') {
    throw new Error('breakdown must be a field name when not using index source');
  }
  return [...(layer.breakdown ? [(0, _columns.getValueColumn)(getAccessorName('breakdown'), layer.breakdown)] : []), (0, _columns.getValueColumn)(ACCESSOR, layer.value, 'number'), ...(layer.queryMaxValue ? [(0, _columns.getValueColumn)(getAccessorName('max'), layer.queryMaxValue, 'number')] : []), ...(layer.querySecondaryMetric ? [(0, _columns.getValueColumn)(getAccessorName('secondary'), layer.querySecondaryMetric)] : [])];
}
async function buildMetric(config, {
  dataViewsAPI
}) {
  const dataviews = {};
  const _buildFormulaLayer = (cfg, i, dataView) => buildFormulaLayer(cfg, i, dataView);
  const datasourceStates = await (0, _utils.buildDatasourceStates)(config, dataviews, _buildFormulaLayer, getValueColumns, dataViewsAPI);
  const {
    references,
    internalReferences,
    adHocDataViews
  } = (0, _utils.extractReferences)(dataviews);
  return {
    title: config.title,
    visualizationType: 'lnsMetric',
    references,
    state: {
      datasourceStates,
      internalReferences,
      filters: [],
      query: {
        language: 'kuery',
        query: ''
      },
      visualization: buildVisualizationState(config),
      adHocDataViews
    }
  };
}