"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.compileTemplate = compileTemplate;
exports.getMetaVariables = getMetaVariables;
var _handlebars = _interopRequireDefault(require("@kbn/handlebars"));
var _jsYaml = require("js-yaml");
var _errors = require("../../../../common/errors");
var _secrets = require("../../secrets");
var _errors2 = require("../../../errors");
var _ = require("../..");
var _cache = require("../packages/cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const handlebars = _handlebars.default.create();
function getMetaVariables(pkg, input, stream) {
  return {
    // Package variables
    package: {
      name: pkg.name,
      title: pkg.title,
      version: pkg.version
    },
    // Stream meta variables
    stream: {
      id: (stream === null || stream === void 0 ? void 0 : stream.id) || '',
      data_stream: {
        dataset: (stream === null || stream === void 0 ? void 0 : stream.data_stream.dataset) || '',
        type: (stream === null || stream === void 0 ? void 0 : stream.data_stream.type) || ''
      }
    },
    // Input meta variables
    input: {
      id: (input === null || input === void 0 ? void 0 : input.id) || ''
    }
  };
}
function compileTemplate(variables, metaVariable, templateStr) {
  const logger = _.appContextService.getLogger();
  const {
    vars,
    yamlValues
  } = buildTemplateVariables(logger, variables, metaVariable);
  let compiledTemplate;
  try {
    let template = (0, _cache.getHandlebarsCompiledTemplateCache)(templateStr);
    if (!template) {
      template = handlebars.compileAST(templateStr, {
        noEscape: true
      });
      (0, _cache.setHandlebarsCompiledTemplateCache)(templateStr, template);
    }
    compiledTemplate = template(vars);
  } catch (err) {
    throw new _errors2.PackageInvalidArchiveError(`Error while compiling agent template: ${err.message}`);
  }
  compiledTemplate = replaceRootLevelYamlVariables(yamlValues, compiledTemplate);
  try {
    const yamlFromCompiledTemplate = (0, _jsYaml.load)(compiledTemplate, {});

    // Hack to keep empty string ('') values around in the end yaml because
    // `load` replaces empty strings with null
    const patchedYamlFromCompiledTemplate = Object.entries(yamlFromCompiledTemplate).reduce((acc, [key, value]) => {
      if (value === null && typeof vars[key] === 'string' && vars[key].trim() === '') {
        acc[key] = '';
      } else {
        acc[key] = value;
      }
      return acc;
    }, {});
    return replaceVariablesInYaml(yamlValues, patchedYamlFromCompiledTemplate);
  } catch (error) {
    const errorMessage = handleYamlError(error, compiledTemplate);
    throw new _errors.PackagePolicyValidationError(errorMessage, error);
  }
}
function handleYamlError(err, yaml) {
  if ((err === null || err === void 0 ? void 0 : err.reason) === 'duplicated mapping key') {
    var _err$mark;
    let position = (_err$mark = err.mark) === null || _err$mark === void 0 ? void 0 : _err$mark.position;
    let key = 'unknown';
    // Read key if position is available
    if (position) {
      key = '';
      while (position < yaml.length && yaml.charAt(position) !== ':') {
        key += yaml.charAt(position);
        position++;
      }
    }
    return `YAMLException: Duplicated key "${key}" found in agent policy yaml, please check your yaml variables.`;
  }
  return err.message;
}
function isValidKey(key) {
  return key !== '__proto__' && key !== 'constructor' && key !== 'prototype';
}
function replaceVariablesInYaml(yamlVariables, yaml) {
  if (Object.keys(yamlVariables).length === 0 || !yaml) {
    return yaml;
  }
  Object.entries(yaml).forEach(([key, value]) => {
    if (typeof value === 'object') {
      yaml[key] = replaceVariablesInYaml(yamlVariables, value);
    }
    if (typeof value === 'string' && value in yamlVariables) {
      yaml[key] = yamlVariables[value];
    }
  });
  return yaml;
}
function buildTemplateVariables(logger, variables, metaVariable) {
  const yamlValues = {};
  const vars = Object.entries(variables).reduce((acc, [key, recordEntry]) => {
    // support variables with . like key.patterns
    const keyParts = key.split('.');
    const lastKeyPart = keyParts.pop();
    if (!lastKeyPart || !isValidKey(lastKeyPart)) {
      throw new _errors2.PackageInvalidArchiveError(`Error while compiling agent template: Invalid key ${lastKeyPart}`);
    }
    let varPart = acc;
    for (const keyPart of keyParts) {
      if (!isValidKey(keyPart)) {
        throw new _errors2.PackageInvalidArchiveError(`Error while compiling agent template: Invalid key ${keyPart}`);
      }
      if (!varPart[keyPart]) {
        varPart[keyPart] = {};
      }
      varPart = varPart[keyPart];
    }
    if (recordEntry.type && recordEntry.type === 'yaml') {
      const yamlKeyPlaceholder = `##${key}##`;
      varPart[lastKeyPart] = recordEntry.value ? `"${yamlKeyPlaceholder}"` : null;
      yamlValues[yamlKeyPlaceholder] = recordEntry.value ? (0, _jsYaml.load)(recordEntry.value) : null;
    } else if (recordEntry.value && recordEntry.value.isSecretRef) {
      if (recordEntry.value.ids) {
        varPart[lastKeyPart] = recordEntry.value.ids.map(id => (0, _secrets.toCompiledSecretRef)(id));
      } else {
        varPart[lastKeyPart] = (0, _secrets.toCompiledSecretRef)(recordEntry.value.id);
      }
    } else {
      varPart[lastKeyPart] = recordEntry.value;
    }
    return acc;
  }, {});
  vars._meta = metaVariable;
  return {
    vars,
    yamlValues
  };
}
function containsHelper(item, check, options) {
  if ((Array.isArray(check) || typeof check === 'string') && check.includes(item)) {
    if (options && options.fn) {
      return options.fn(this);
    }
    return true;
  }
  return '';
}
handlebars.registerHelper('contains', containsHelper);

// escapeStringHelper will wrap the provided string with single quotes.
// Single quoted strings in yaml need to escape single quotes by doubling them
// and to respect any incoming newline we also need to double them, otherwise
// they will be replaced with a space.
function escapeStringHelper(str) {
  if (!str) return undefined;
  return "'" + str.replace(/\'/g, "''").replace(/\n/g, '\n\n') + "'";
}
handlebars.registerHelper('escape_string', escapeStringHelper);

/**
 * escapeMultilineStringHelper will escape a multiline string by doubling the newlines
 * and escaping single quotes.
 * This is useful when the string is multiline and needs to be escaped in a yaml file
 * without wrapping it in single quotes.
 */
function escapeMultilineStringHelper(str) {
  if (!str) return undefined;
  return str.replace(/\'/g, "''").replace(/\n/g, '\n\n');
}
handlebars.registerHelper('escape_multiline_string', escapeMultilineStringHelper);

// toJsonHelper will convert any object to a Json string.
function toJsonHelper(value) {
  if (typeof value === 'string') {
    // if we get a string we assume is an already serialized json
    return value;
  }
  return JSON.stringify(value);
}
handlebars.registerHelper('to_json', toJsonHelper);

// urlEncodeHelper returns a string encoded as a URI component.
function urlEncodeHelper(input) {
  let encodedString = encodeURIComponent(input);
  // encodeURIComponent does not encode the characters -.!~*'(), known as "unreserved marks",
  // which do not have a reserved purpose but are allowed in a URI "as is". So, these have are
  // explicitly encoded. The following creates the sequences %27 %28 %29 %2A. Since the valid
  // encoding of "*" is %2A, it is necessary to call toUpperCase() to properly encode.
  encodedString = encodedString.replace(/[!'()*]/g, char => '%' + char.charCodeAt(0).toString(16).toUpperCase());
  return encodedString;
}
handlebars.registerHelper('url_encode', urlEncodeHelper);
function replaceRootLevelYamlVariables(yamlVariables, yamlTemplate) {
  if (Object.keys(yamlVariables).length === 0 || !yamlTemplate) {
    return yamlTemplate;
  }
  let patchedTemplate = yamlTemplate;
  Object.entries(yamlVariables).forEach(([key, val]) => {
    patchedTemplate = patchedTemplate.replace(new RegExp(`^"${key}"`, 'gm'), () => val ? (0, _jsYaml.dump)(val) : '');
  });
  return patchedTemplate;
}