"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteEnrollmentApiKey = deleteEnrollmentApiKey;
exports.deleteEnrollmentApiKeyForAgentPolicyId = deleteEnrollmentApiKeyForAgentPolicyId;
exports.ensureDefaultEnrollmentAPIKeyForAgentPolicy = ensureDefaultEnrollmentAPIKeyForAgentPolicy;
exports.generateEnrollmentAPIKey = generateEnrollmentAPIKey;
exports.getEnrollmentAPIKey = getEnrollmentAPIKey;
exports.getEnrollmentAPIKeyById = getEnrollmentAPIKeyById;
exports.hasEnrollementAPIKeysForPolicy = hasEnrollementAPIKeysForPolicy;
exports.listEnrollmentApiKeys = listEnrollmentApiKeys;
var _uuid = require("uuid");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _i18n = require("@kbn/i18n");
var _elasticsearch = require("@elastic/elasticsearch");
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/spaces-plugin/common");
var _constants = require("../../../common/constants");
var _errors = require("../../errors");
var _constants2 = require("../../constants");
var _agent_policy = require("../agent_policy");
var _saved_object = require("../saved_object");
var _audit_logging = require("../audit_logging");
var _agents = require("../agents");
var _app_context = require("../app_context");
var _helpers = require("../spaces/helpers");
var _agent_namespaces = require("../spaces/agent_namespaces");
var _security = require("./security");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const uuidRegex = /^\([0-9a-fA-F]{8}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{12}\)$/;
async function listEnrollmentApiKeys(esClient, options) {
  const {
    page = 1,
    perPage = 20,
    kuery,
    spaceId
  } = options;
  // const query = options.query ?? (kuery && toElasticsearchQuery(fromKueryExpression(kuery)));

  let query;
  if (options.query && spaceId) {
    throw new Error('not implemented (query should only be used in Fleet internal usage)');
  }
  if (!options.query) {
    const filters = [];
    if (kuery) {
      filters.push(kuery);
    }
    const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
    if (useSpaceAwareness && spaceId) {
      if (spaceId === _common.DEFAULT_SPACE_ID) {
        filters.push(_agent_namespaces.DEFAULT_NAMESPACES_FILTER);
      } else {
        filters.push(`namespaces:"${spaceId}" or namespaces:"${_constants.ALL_SPACES_ID}"`);
      }
    }
    const kueryNode = (0, _agents._joinFilters)(filters);
    query = kueryNode ? (0, _esQuery.toElasticsearchQuery)(kueryNode) : undefined;
  } else {
    query = options.query;
  }
  const res = await esClient.search({
    index: _constants2.ENROLLMENT_API_KEYS_INDEX,
    from: (page - 1) * perPage,
    size: perPage,
    track_total_hits: true,
    rest_total_hits_as_int: true,
    ignore_unavailable: true,
    sort: [{
      created_at: {
        order: 'desc'
      }
    }],
    ...(query ? {
      query
    } : {})
  });

  // @ts-expect-error @elastic/elasticsearch _source is optional
  const items = res.hits.hits.map(esDocToEnrollmentApiKey);
  return {
    items,
    total: res.hits.total,
    page,
    perPage
  };
}
async function hasEnrollementAPIKeysForPolicy(esClient, policyId) {
  const res = await listEnrollmentApiKeys(esClient, {
    kuery: `policy_id:"${policyId}"`
  });
  return res.total !== 0;
}
async function getEnrollmentAPIKey(esClient, id, spaceId) {
  try {
    const body = await esClient.get({
      index: _constants2.ENROLLMENT_API_KEYS_INDEX,
      id
    });
    if (spaceId) {
      var _body$_source, _body$_source$namespa, _body$_source4, _body$_source4$namesp;
      if ((_body$_source = body._source) !== null && _body$_source !== void 0 && (_body$_source$namespa = _body$_source.namespaces) !== null && _body$_source$namespa !== void 0 && _body$_source$namespa.includes(_constants.ALL_SPACES_ID)) {
        // Do nothing all spaces have access to this key
      } else if (spaceId === _common.DEFAULT_SPACE_ID) {
        var _body$_source$namespa2, _body$_source2, _body$_source2$namesp, _body$_source3, _body$_source3$namesp;
        if (((_body$_source$namespa2 = (_body$_source2 = body._source) === null || _body$_source2 === void 0 ? void 0 : (_body$_source2$namesp = _body$_source2.namespaces) === null || _body$_source2$namesp === void 0 ? void 0 : _body$_source2$namesp.length) !== null && _body$_source$namespa2 !== void 0 ? _body$_source$namespa2 : 0) > 0 && !((_body$_source3 = body._source) !== null && _body$_source3 !== void 0 && (_body$_source3$namesp = _body$_source3.namespaces) !== null && _body$_source3$namesp !== void 0 && _body$_source3$namesp.includes(_common.DEFAULT_SPACE_ID))) {
          throw new _errors.EnrollmentKeyNotFoundError(`Enrollment api key ${id} not found in namespace`);
        }
      } else if (!((_body$_source4 = body._source) !== null && _body$_source4 !== void 0 && (_body$_source4$namesp = _body$_source4.namespaces) !== null && _body$_source4$namesp !== void 0 && _body$_source4$namesp.includes(spaceId))) {
        throw new _errors.EnrollmentKeyNotFoundError(`Enrollment api key ${id} not found in namespace`);
      }
    }

    // @ts-expect-error esDocToEnrollmentApiKey doesn't accept optional _source
    return esDocToEnrollmentApiKey(body);
  } catch (e) {
    if (e instanceof _elasticsearch.errors.ResponseError && e.statusCode === 404) {
      throw new _errors.EnrollmentKeyNotFoundError(`Enrollment api key ${id} not found`);
    }
    throw e;
  }
}

/**
 * Invalidate an api key and mark it as inactive
 * @param id
 */
async function deleteEnrollmentApiKey(esClient, id, forceDelete = false, spaceId) {
  const logger = _app_context.appContextService.getLogger();
  logger.debug(`Deleting enrollment API key ${id}`);
  const enrollmentApiKey = await getEnrollmentAPIKey(esClient, id, spaceId);
  _audit_logging.auditLoggingService.writeCustomAuditLog({
    message: `User deleting enrollment API key [id=${enrollmentApiKey.id}] [api_key_id=${enrollmentApiKey.api_key_id}]`
  });
  await (0, _security.invalidateAPIKeys)([enrollmentApiKey.api_key_id]);
  if (forceDelete) {
    await esClient.delete({
      index: _constants2.ENROLLMENT_API_KEYS_INDEX,
      id,
      refresh: 'wait_for'
    });
  } else {
    await esClient.update({
      index: _constants2.ENROLLMENT_API_KEYS_INDEX,
      id,
      doc: {
        active: false
      },
      refresh: 'wait_for'
    });
  }
  logger.debug(`Deleted enrollment API key ${enrollmentApiKey.id} [api_key_id=${enrollmentApiKey.api_key_id}`);
}
async function deleteEnrollmentApiKeyForAgentPolicyId(esClient, agentPolicyId) {
  let hasMore = true;
  let page = 1;
  while (hasMore) {
    const {
      items
    } = await listEnrollmentApiKeys(esClient, {
      page: page++,
      perPage: 100,
      kuery: `policy_id:"${agentPolicyId}"`
    });
    if (items.length === 0) {
      hasMore = false;
    }
    for (const apiKey of items) {
      await deleteEnrollmentApiKey(esClient, apiKey.id);
    }
  }
}
async function generateEnrollmentAPIKey(soClient, esClient, data) {
  const id = (0, _uuid.v4)();
  const {
    name: providedKeyName,
    forceRecreate,
    agentPolicyId
  } = data;
  const logger = _app_context.appContextService.getLogger();
  logger.debug(`Creating enrollment API key ${JSON.stringify(data)}`);
  const agentPolicy = await retrieveAgentPolicyId(soClient, agentPolicyId);
  if (providedKeyName && !forceRecreate) {
    let hasMore = true;
    let page = 1;
    let keys = [];
    while (hasMore) {
      const {
        items
      } = await listEnrollmentApiKeys(esClient, {
        page: page++,
        perPage: 100,
        query: getQueryForExistingKeyNameOnPolicy(agentPolicyId, providedKeyName)
      });
      if (items.length === 0) {
        hasMore = false;
      } else {
        keys = keys.concat(items);
      }
    }
    if (keys.length > 0 && keys.some(k => {
      var _k$name;
      return (// Prevent false positives when the providedKeyName is a prefix of a token name that already exists
        // After removing the providedKeyName and trimming whitespace, the only string left should be a uuid in parens.
        (_k$name = k.name) === null || _k$name === void 0 ? void 0 : _k$name.replace(providedKeyName, '').trim().match(uuidRegex)
      );
    })) {
      throw new _errors.EnrollmentKeyNameExistsError(_i18n.i18n.translate('xpack.fleet.serverError.enrollmentKeyDuplicate', {
        defaultMessage: 'An enrollment key named {providedKeyName} already exists for agent policy {agentPolicyId}',
        values: {
          providedKeyName,
          agentPolicyId
        }
      }));
    }
  }
  const name = providedKeyName ? `${providedKeyName} (${id})` : id;
  _audit_logging.auditLoggingService.writeCustomAuditLog({
    message: `User creating enrollment API key [name=${name}] [policy_id=${agentPolicyId}]`
  });
  logger.debug(`Creating enrollment API key [name=${name}] [policy_id=${agentPolicyId}]`);
  const key = await esClient.security.createApiKey({
    name,
    metadata: {
      managed_by: 'fleet',
      managed: true,
      type: 'enroll',
      policy_id: data.agentPolicyId
    },
    role_descriptors: {
      // Useless role to avoid to have the privilege of the user that created the key
      'fleet-apikey-enroll': {
        cluster: [],
        index: [],
        applications: [{
          application: 'fleet',
          privileges: ['no-privileges'],
          resources: ['*']
        }]
      }
    }
  }).catch(err => {
    throw new _errors.FleetError(`Impossible to create an api key: ${err.message}`);
  });
  if (!key) {
    throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.serverError.unableToCreateEnrollmentKey', {
      defaultMessage: 'Unable to create an enrollment api key'
    }));
  }
  const apiKey = Buffer.from(`${key.id}:${key.api_key}`).toString('base64');
  const body = {
    active: true,
    api_key_id: key.id,
    api_key: apiKey,
    name,
    policy_id: agentPolicyId,
    namespaces: agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.space_ids,
    created_at: new Date().toISOString(),
    hidden: (agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.supports_agentless) || (agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.is_managed)
  };
  const res = await esClient.create({
    index: _constants2.ENROLLMENT_API_KEYS_INDEX,
    body,
    id,
    refresh: 'wait_for'
  });
  const enrollmentAPIKey = {
    id: res._id,
    api_key_id: body.api_key_id,
    api_key: body.api_key,
    name: body.name,
    active: body.active,
    policy_id: body.policy_id,
    created_at: body.created_at
  };
  return enrollmentAPIKey;
}
async function ensureDefaultEnrollmentAPIKeyForAgentPolicy(soClient, esClient, agentPolicyId) {
  const hasKey = await hasEnrollementAPIKeysForPolicy(esClient, agentPolicyId);
  if (hasKey) {
    return;
  }
  return generateEnrollmentAPIKey(soClient, esClient, {
    name: `Default`,
    agentPolicyId,
    forceRecreate: true // Always generate a new enrollment key when Fleet is being set up
  });
}
function getQueryForExistingKeyNameOnPolicy(agentPolicyId, providedKeyName) {
  const query = {
    bool: {
      filter: [{
        bool: {
          should: [{
            match_phrase: {
              policy_id: agentPolicyId
            }
          }],
          minimum_should_match: 1
        }
      }, {
        bool: {
          should: [{
            query_string: {
              fields: ['name'],
              query: `(${providedKeyName.replace('!', '\\!')}) *`
            }
          }],
          minimum_should_match: 1
        }
      }]
    }
  };
  return query;
}
async function getEnrollmentAPIKeyById(esClient, apiKeyId) {
  const res = await esClient.search({
    index: _constants2.ENROLLMENT_API_KEYS_INDEX,
    ignore_unavailable: true,
    q: `api_key_id:${(0, _saved_object.escapeSearchQueryPhrase)(apiKeyId)}`
  });

  // @ts-expect-error esDocToEnrollmentApiKey doesn't accept optional _source
  const [enrollmentAPIKey] = res.hits.hits.map(esDocToEnrollmentApiKey);
  if ((enrollmentAPIKey === null || enrollmentAPIKey === void 0 ? void 0 : enrollmentAPIKey.api_key_id) !== apiKeyId) {
    throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.serverError.returnedIncorrectKey', {
      defaultMessage: 'Find enrollmentKeyById returned an incorrect key'
    }));
  }
  return enrollmentAPIKey;
}
async function retrieveAgentPolicyId(soClient, agentPolicyId) {
  return _agent_policy.agentPolicyService.get(soClient, agentPolicyId).catch(async e => {
    if (e.isBoom && e.output.statusCode === 404) {
      throw _boom.default.badRequest(_i18n.i18n.translate('xpack.fleet.serverError.agentPolicyDoesNotExist', {
        defaultMessage: 'Agent policy {agentPolicyId} does not exist',
        values: {
          agentPolicyId
        }
      }));
    }
    throw e;
  });
}
function esDocToEnrollmentApiKey(doc) {
  return {
    id: doc._id,
    api_key_id: doc._source.api_key_id,
    api_key: doc._source.api_key,
    name: doc._source.name,
    policy_id: doc._source.policy_id,
    created_at: doc._source.created_at,
    active: doc._source.active || false,
    hidden: doc._source.hidden || false
  };
}