"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.correctQuerySyntax = correctQuerySyntax;
exports.findAstPosition = findAstPosition;
exports.getBracketsToClose = getBracketsToClose;
exports.isMarkerNode = isMarkerNode;
exports.isNotMarkerNodeOrArray = isNotMarkerNodeOrArray;
exports.mapToNonMarkerNode = mapToNonMarkerNode;
exports.removeMarkerArgFromArgsList = removeMarkerArgFromArgsList;
var _constants = require("../constants");
var _is = require("../../ast/is");
var _ = require("../../..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function isMarkerNode(node) {
  if (Array.isArray(node)) {
    return false;
  }
  return Boolean(node && ((0, _is.isColumn)(node) || (0, _is.isIdentifier)(node) || (0, _is.isSource)(node)) && node.name.endsWith(_constants.EDITOR_MARKER));
}
function findCommand(ast, offset) {
  const queryCommands = ast.commands;
  const commandIndex = queryCommands.findIndex(({
    location
  }) => location.min <= offset && location.max >= offset);
  const command = queryCommands[commandIndex] || queryCommands[queryCommands.length - 1];
  if (!command) {
    return findHeaderCommand(ast, offset);
  }
  return command;
}
function findHeaderCommand(ast, offset) {
  if (!ast.header || ast.header.length === 0) {
    return;
  }
  const commandIndex = ast.header.findIndex(({
    location
  }) => location.min <= offset && location.max >= offset);
  const targetHeader = ast.header[commandIndex] || ast.header[ast.header.length - 1];
  return targetHeader.incomplete ? targetHeader : undefined;
}
function isNotMarkerNodeOrArray(arg) {
  return Array.isArray(arg) || !isMarkerNode(arg);
}
const removeMarkerNode = node => {
  _.Walker.walk(node, {
    visitAny: current => {
      if ('args' in current) {
        current.args = current.args.filter(n => !isMarkerNode(n));
      } else if ('values' in current) {
        current.values = current.values.filter(n => !isMarkerNode(n));
      }
    }
  });
};
function removeMarkerArgFromArgsList(node) {
  if (!node) {
    return;
  }
  if (node.type === 'command' || node.type === 'option' || node.type === 'function') {
    return {
      ...node,
      args: node.args.filter(isNotMarkerNodeOrArray).map(mapToNonMarkerNode)
    };
  }
  return node;
}
function mapToNonMarkerNode(arg) {
  return Array.isArray(arg) ? arg.filter(isNotMarkerNodeOrArray).map(mapToNonMarkerNode) : arg;
}
function cleanMarkerNode(node) {
  return isMarkerNode(node) ? undefined : node;
}
function findOption(nodes, offset) {
  return findCommandSubType(nodes, offset, _is.isOptionNode);
}
function findCommandSubType(nodes, offset, isOfTypeFn) {
  for (const node of nodes) {
    if (isOfTypeFn(node)) {
      if (node.location.min <= offset && node.location.max >= offset || nodes[nodes.length - 1] === node && node.location.max < offset) {
        return node;
      }
    }
  }
}
function findAstPosition(ast, offset) {
  const command = findCommand(ast, offset);
  if (!command) {
    return {
      command: undefined,
      node: undefined
    };
  }
  let containingFunction;
  let node;
  _.Walker.walk(command, {
    visitSource: (_node, parent, walker) => {
      if (_node.location.max >= offset && _node.text !== _constants.EDITOR_MARKER) {
        node = _node;
        walker.abort();
      }
    },
    visitAny: _node => {
      var _node$location;
      if (_node.type === 'function' && _node.subtype === 'variadic-call' && ((_node$location = _node.location) === null || _node$location === void 0 ? void 0 : _node$location.max) >= offset) {
        containingFunction = _node;
      }
      if (_node.location.max >= offset && _node.text !== _constants.EDITOR_MARKER && (!(0, _is.isList)(_node) || _node.subtype !== 'tuple')) {
        node = _node;
      }
    }
  });
  if (node) removeMarkerNode(node);
  return {
    command: removeMarkerArgFromArgsList(command),
    containingFunction: removeMarkerArgFromArgsList(containingFunction),
    option: removeMarkerArgFromArgsList(findOption(command.args, offset)),
    node: removeMarkerArgFromArgsList(cleanMarkerNode(node))
  };
}

/**
 * This function returns a list of closing brackets that can be appended to
 * a partial query to make it valid.
 *
 * A known limitation of this is that is not aware of commas "," or pipes "|"
 * so it is not yet helpful on a multiple commands errors (a workaround is to pass each command here...)
 * @param text
 * @returns
 */
function getBracketsToClose(text) {
  const stack = [];
  const pairs = {
    '"""': '"""',
    '/*': '*/',
    '(': ')',
    '[': ']',
    '"': '"'
  };
  const pairsReversed = {
    '"""': '"""',
    '*/': '/*',
    ')': '(',
    ']': '[',
    '"': '"'
  };
  for (let i = 0; i < text.length; i++) {
    const isInsideString = stack.some(item => item === '"' || item === '"""');
    for (const openBracket in pairs) {
      if (!Object.hasOwn(pairs, openBracket)) {
        continue;
      }
      const substr = text.slice(i, i + openBracket.length);

      // Skip comment markers (/* and */) when inside a string
      if (isInsideString && (openBracket === '/*' || substr === '*/')) {
        continue;
      }
      if (pairsReversed[substr] && pairsReversed[substr] === stack[stack.length - 1]) {
        stack.pop();
        break;
      } else if (substr === openBracket) {
        stack.push(substr);
        break;
      }
    }
  }
  return stack.reverse().map(bracket => pairs[bracket]);
}

/**
 * This function attempts to correct the syntax of a partial query to make it valid.
 *
 * We are generally dealing with incomplete queries when the user is typing. But,
 * having an AST is helpful so we heuristically correct the syntax so it can be parsed.
 *
 * @param _query
 * @param context
 * @returns
 */
function correctQuerySyntax(_query) {
  let query = _query;
  // check if all brackets are closed, otherwise close them
  const bracketsToAppend = getBracketsToClose(query);
  const endsWithBinaryOperatorRegex = /(?:\+|\/|==|>=|>|in|<=|<|like|:|%|\*|-|not in|not like|not rlike|!=|rlike|and|or|not|=|as)\s+$/i;
  const endsWithCommaRegex = /,\s+$/;
  if (endsWithBinaryOperatorRegex.test(query) || endsWithCommaRegex.test(query)) {
    query += ` ${_constants.EDITOR_MARKER}`;
  }
  query += bracketsToAppend.join('');
  return query;
}