"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getGeneration = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves an attack discovery generation by execution UUID with error handling.
 *
 * If the generation metadata is not found (404) but alert data exists, it returns undefined
 * to allow the response to proceed with just the alert data.
 *
 * @returns Promise<AttackDiscoveryGeneration | undefined> - The generation metadata or undefined if not found but data exists
 * @throws Error for any errors other than 404 with existing data
 */
const getGeneration = async ({
  dataClient,
  authenticatedUser,
  eventLogIndex,
  executionUuid,
  logger,
  spaceId,
  data
}) => {
  try {
    return await dataClient.getAttackDiscoveryGenerationById({
      authenticatedUser,
      eventLogIndex,
      executionUuid,
      logger,
      spaceId
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    // If generation metadata is not found (404) but we have data, continue with undefined generation
    if (error.statusCode === 404 && data.length > 0) {
      logger.debug(() => `Generation metadata not found for execution_uuid ${executionUuid}, but returning ${data.length} discovery alerts`);
      return undefined;
    } else {
      // Re-throw other errors or 404 when there's no data
      throw err;
    }
  }
};
exports.getGeneration = getGeneration;